package org.eu.net.pool.mutationkit

import com.jamieswhiteshirt.reachentityattributes.ReachEntityAttributes
import com.mojang.serialization.Codec
import net.minecraft.entity.attribute.{EntityAttribute, EntityAttributeModifier, EntityAttributes}
import net.minecraft.entity.player.PlayerEntity
import net.minecraft.util.Identifier

import java.util.UUID

given Codec[Double] = Codec.DOUBLE.xmap(identity(_), identity(_))
given Codec[Identifier] = Identifier.CODEC
given Codec[Unit] = Codec.unit(())

trait AttributeMutationType(name: String, attributes: (EntityAttribute, UUID)*) extends MutationType[Double]:
  override def onApplied(player: PlayerEntity, data: Double): Unit =
    for (attr, id) <- attributes do
      player.getAttributes.getCustomInstance(attr).addPersistentModifier(EntityAttributeModifier(id, s"$name mutation", data, EntityAttributeModifier.Operation.ADDITION))
  override def onRemoved(player: PlayerEntity, data: Double): Unit =
    for (attr, id) <- attributes do
      player.getAttributes.getCustomInstance(attr).removeModifier(id)

object MaxHealth extends AttributeMutationType("Max health", EntityAttributes.GENERIC_MAX_HEALTH -> UUID(-9145582182894646066L, -5921583488935989421L)):
  override def conflictSet = Set(Reach)
object Armor extends AttributeMutationType("Armor", EntityAttributes.GENERIC_ARMOR -> UUID(8831851519878710740L, -7367767561233476871L)):
  override def conflictSet = Set(MovementSpeed)
object AttackPower extends AttributeMutationType("Attack power", EntityAttributes.GENERIC_ATTACK_DAMAGE -> UUID(-3692264523689147600L, -5791999230100230385L), EntityAttributes.GENERIC_ATTACK_KNOCKBACK -> UUID(4200557496438965762L, -5420775957259662849L)):
  override def conflictSet = Set(AttackSpeed)
object AttackSpeed extends AttributeMutationType("Attack speed", EntityAttributes.GENERIC_ATTACK_SPEED -> UUID(4900495767084908233L, 7037697696485768230L)):
  override def conflictSet = Set(AttackPower)
object KnockbackResistance extends AttributeMutationType("Knockback resistance", EntityAttributes.GENERIC_KNOCKBACK_RESISTANCE -> UUID(7768237887541238230L, -7037697696485768230L)):
  override def conflictSet = Set(MovementSpeed)
object MovementSpeed extends AttributeMutationType("Movement speed", EntityAttributes.GENERIC_MOVEMENT_SPEED -> UUID(1234567890123456789L, -1234567890123456789L)):
  override def conflictSet = Set(Armor, KnockbackResistance, SwimSpeed)
object Reach extends AttributeMutationType("Reach", ReachEntityAttributes.REACH -> UUID(987654321098765432L, -987654321098765432L), ReachEntityAttributes.ATTACK_RANGE -> UUID(-8723051609705265141L, -7354226922437916459L)):
  override def conflictSet = Set(MaxHealth)
object StepHeight extends MutationType[Double]:
  override def conflictSet = Set(JumpPower, SwimSpeed)
object JumpPower extends MutationType[Double]:
  override def conflictSet = Set(StepHeight, SwimSpeed)
object Gravity extends MutationType[Double]
object SwimSpeed extends MutationType[Double]: // TODO
  override def conflictSet = Set(MovementSpeed, Spiderlike)

object Spiderlike extends MutationType[Unit]:
  override def conflictSet = Set(StepHeight, JumpPower, SwimSpeed, Frictionless)
object Volatile extends MutationType[Unit]: // TODO
  override def conflictSet = Set(Breathless, Conductive)
object Frictionless extends MutationType[Unit]: // TODO
  override def conflictSet = Set(Spiderlike)
object Breathless extends MutationType[Unit]: // TODO
  override def conflictSet = Set(Volatile)
object Glowing extends MutationType[Unit]:
  override def conflictSet = Set(LightSensitive)
object VisionAnomaly extends MutationType[Identifier]: // TODO
  override def conflictSet = Set(LightSensitive, Schizo)
object Conductive extends MutationType[Unit]: // TODO
  override def conflictSet = Set(Volatile)
object LightSensitive extends MutationType[Unit]: // TODO
  override def conflictSet = Set(Glowing, VisionAnomaly)
object Schizo extends MutationType[Unit]: // TODO
  override def conflictSet = Set(LightSensitive)