package org.net.eu.pool.mica

import com.mojang.serialization.{Codec, Lifecycle}
import net.fabricmc.fabric.api.event.registry.FabricRegistryBuilder
import net.minecraft.registry.{MutableRegistry, Registries, Registry, RegistryKey, SimpleRegistry}
import net.minecraft.util.Identifier

import scala.annotation.meta.companionObject
import scala.annotation.{MacroAnnotation, compileTimeOnly, experimental}
import scala.collection.mutable
import scala.compiletime.summonInline
import scala.deriving.Mirror
import scala.language.experimental.macros
import scala.quoted.{Expr, Quotes, ToExpr, Type}
import scala.reflect.ClassTag

/**
 * Helper type used with [[register]] for setting your mod ID.
 * @param name Mod ID, pursuant to identifier rules.
 */
case class ModID(name: String):
  Identifier.of(name, "any")

/**
 * Returns the mod ID of the current file.
 */
inline def modid(using m: ModID): String = m.name

/**
 * '''Deprecated''': The generated givens are hard to use in code. Derive [[HasRegistry]] instead.
 *
 * This macro generates a registry for a class, allowing the [[register @register]] annotation to track its descendants.
 */
@deprecated("@hasRegistry is hard to use correctly. Opt to derive HasRegistry instead.")
@compileTimeOnly("@hasRegistry is expanded at compile-time")
class hasRegistry extends MacroAnnotation:
  // Macro-class: we receive the `Definition` of what we're applied as well as its companion object (if it has one).
  override def transform(using q: Quotes)(definition: q.reflect.Definition, companion: Option[q.reflect.Definition]): List[q.reflect.Definition] =
    import q.reflect.*
    // First, match on `definition` - we only support generating registries for classes.
    definition match
      // if it *is* a class, extract its name and also save the class itself to `cl`.
      case cl@ClassDef(name, _, _, _, _) =>
        // Next, match on its companion object:
        companion match
          // In our case, we require it to exist and to be a class, since we plan to add givens to it.
          // Save everything for later, since we'll remake it in our return value.
          case Some(cd@ClassDef(cname, constr, parents, self, body)) =>
            // Prepare the list of the statements we want to generate:
            val givens =
              // Typematch (extract the *actual type* of an expression) the class.
              cl.symbol.typeRef.asType match
                // This unconditionally brings the class itself into scope as `t` - we know nothing about it, not even
                //  whether it's an object, but we don't need to know anything to make a registry for it.
                case '[t] =>
                  // Construct `Symbol`s (which are basically fancy variable names, but they also store the type and associated metadata)
                  //  for the registry and its friends. In this case, all the variables are marked as `given` (making them available for
                  //  `summon` to use), as well as synthetic (since they're generated by a macro). The actual names aren't important assuming
                  //  they're unique, so I use a name mirroring the ones Scala generates for unnamed givens. The first `owner` parameter is
                  //  set to the companion object, as they must be - we later append them to the real companion object, so Scala will error
                  //  otherwise. The *final* parameter sets the visibility of the symbol: Scala visibilities are different from Java's four
                  //  visibilities, in that Scala visibilities are based on an *enclosing* type... I'm getting off-topic. Anyway.
                  // Construct three Symbols representing three implicit variables for Registry-related stuff.
                  val keySym = Symbol.newVal(cd.symbol, s"registryKey", TypeRepr.of[RegistryKey[Registry[t]]], Flags.Given | Flags.Synthetic, Symbol.noSymbol)
                  val registrySym = Symbol.newVal(cd.symbol, s"registry", TypeRepr.of[MutableRegistry[t]], Flags.Given | Flags.Synthetic, Symbol.noSymbol)
                  val codecSym = Symbol.newVal(cd.symbol, s"registryCodec", TypeRepr.of[Codec[t]], Flags.Given | Flags.Synthetic, Symbol.noSymbol)
                  // Generate a list of the actual registry stuff definitions.
                  Seq(
                    ValDef(keySym, Some {
                      // ...find a mod ID...
                      val modID = Expr.summon[ModID].getOrElse:
                        report.error("No mod ID present in file")
                        '{ ??? }
                      // ...make a RegistryKey from it...
                      val regKey: Expr[RegistryKey[Registry[t]]] = '{RegistryKey.ofRegistry[t](Identifier.of(${modID}.name, ${Expr(name.toLowerCase)}))}
                      // ...and convert it to an untyped Term for our mutilation.
                      regKey.asTerm
                    }),
                    ValDef(registrySym, Some {
                      // Similar to before, but different:
                      // pull out the keySym definition from before, casting it to the type we know it is...
                      val keyExpr: Expr[RegistryKey[Registry[t]]] = Ref(keySym).asExprOf[RegistryKey[Registry[t]]]
                      // ...use Fabric's API to create and register a registry from it...
                      val registryExpr = '{FabricRegistryBuilder.createSimple(${keyExpr}).buildAndRegister()}
                      // ...and nuke type-safety again.
                      registryExpr.asTerm
                    }),
                    ValDef(codecSym, Some {
                      // This is left as an exercise to the reader.
                      '{${Ref(registrySym).asExprOf[MutableRegistry[t]]}.getCodec}.asTerm
                    }),
                  )
            // Return the definition, and a copy of the companion object plus our appended givens.
            List(definition, ClassDef.copy(cd)(cname, constr, parents, self, body ++ givens))
          // Of course, being a macro, we can't simply fail if it doesn't exist! We need to handle this case too.
          // Ideally, we'd handle this case by synthesizing a companion of our own, but I'm too lazy to do that.
          case None =>
            // Get the class's name-if we know what it is. If we do, pass it to report.error, or if not, call it with no position (blaming it on the macro call as a whole).
            cl.symbol.pos.fold(report.error("Missing companion object for @hasRegistry"))(report.error("Missing companion object for @hasRegistry", _))
            // and, since we *must* return, just return the original definition unmodified.
            List(definition)
      // and handle someone applying this to e.g. a function of course, since of course someone will.
      case _ =>
        report.error("Only class definitions may have associated registries")
        List(definition) :++ companion

/**
 * Adds a hook to when a given object is registered by [[register @register]]. This could be used to, for example, add registrations to other registries (as [[Rune]] does).
 * @tparam T Object to listen for registrations on. This should be the *concrete type* of your object, not of the registry!
 */
private[mica] trait RegisterHook[T]:
  def preRegister(target: T, key: Identifier)(using registry: Registry[? >: T]): Boolean = true
  def postRegister(target: T, key: Identifier)(using registry: Registry[? >: T]): Unit = ()

private var registrarState: Option[mutable.Buffer[Expr[Unit]]] = Some(mutable.Buffer.empty)

/**
 * Registers the given value into the best registry for it. This macro may only be applied to `object` literals.
 * @param key Combined with a [[ModID]] in scope to create the value's [[Identifier]]
 */
@compileTimeOnly("@register is expanded at compile-time")
class register(key: String) extends MacroAnnotation:
  override def transform(using q: Quotes)(definition: q.reflect.Definition, companion: Option[q.reflect.Definition]): List[q.reflect.Definition] =
    import q.reflect.*
    val cl =
      definition match
        case cl@ClassDef(_, _, _, _, _) if cl.symbol.flags.is(Flags.Module) => cl
        case _ =>
          report.error("Only objects may be annotated at this time")
          return List(definition) :++ companion
    cl.symbol.typeRef.asType match
      case '[t] =>
        registrarState match
          case Some(l) =>
            val modid: Expr[ModID] = Expr.summon[ModID]
              .getOrElse:
                report.error(s"No mod ID found in file")
                '{ ModID("unknown") }
            val actualThis: Expr[t] = Ref(cl.symbol.companionModule).asExprOf[t]
            l += '{
              Registry.register(summonInline[Registry[? >: t]], Identifier.of(${modid}.name, ${Expr(key)}), ${actualThis})
              ()
            }
            List(ClassDef.copy(cl)(cl.symbol.name, cl.constructor, cl.parents, cl.self, cl.body)) :++ companion
          case None =>
            report.error("Registrations were processed too early")
    List(definition) :++ companion
end register
inline def trySummon[T]: Option[T] = compiletime.summonFrom:
  case x: T => Some(x)
  case _ => None
def liftOption[T: Type](expr: Option[Expr[T]])(using Quotes): Expr[Option[T]] = expr match
  case Some(value) => '{Some(${value})}
  case None => '{None}
object register:
  /**
   * '''Implementation detail''' used to track [[register]]-blocks. Do not use.
   * @deprecated Do not use.
   */
  private[register] val _inits = mutable.Map.empty[String, Any]

  /**
   * Executes all registrations created by [[register @register]] annotations and [[apply(R):R* register {...}]] blocks.
   * {{{
   * trait Foo derives HasRegistry
   *
   * @register("foo_impl")
   * object FooImpl extends Foo
   *
   * def init() =
   *   register()
   * }}}
   */
  @compileTimeOnly("@register is expanded at compile-time")
  inline def apply(): Unit = ${ register.apply_impl }

  /**
   * Appends arbitrary code to the registration queue.
   *
   * > [!CAUTION]
   * > If [[value]] references a local scope not visible from the [[apply():Unit* register()]] invocation, the compiler will crash!
   *
   * @param value
   * @tparam R
   * @return
   */
  @compileTimeOnly("@register is expanded at compile-time")
  inline def apply(value: => Unit): Unit = ${ register.apply_impl('value) }
  private def apply_impl(using q: Quotes): Expr[Unit] =
    import q.reflect.*
    registrarState match
      case Some(terms) =>
        registrarState = None
        Expr.block(terms.toList, '{()})
      case None =>
        report.error("Registrations have already been processed")
        '{()}
  end apply_impl
  private def apply_impl(body: Expr[Unit])(using q: Quotes): Expr[Unit] =
    import q.reflect.*
    registrarState match
      case Some(l) =>
        val name = Symbol.freshName("register")
        l += body
      case None =>
        report.error("Registrations have already been processed")
    '{()}

trait HasRegistry[T]:
  given registryKey: RegistryKey[Registry[T]]
  given registry: MutableRegistry[T]
  given registryCodec: Codec[T] = registry.getCodec
object HasRegistry:
  inline def derived[T: ClassTag as tag](using modID: ModID) = new HasRegistry[T]:
    override given registryKey: RegistryKey[Registry[T]] = RegistryKey.ofRegistry(Identifier.of(modID.name, tag.runtimeClass.getName.toLowerCase))
    override given registry: MutableRegistry[T] = FabricRegistryBuilder.createSimple(registryKey).buildAndRegister()

def registryFor[T: MutableRegistry as r]: MutableRegistry[T] = r

// help Scala pull givens out of HasRegistry
given byHavingRegistry: [T: HasRegistry as r] => MutableRegistry[T] = r.registry
given byHavingRegistryKey: [T: HasRegistry as r] => RegistryKey[Registry[T]] = r.registryKey
given byHavingRegistryCodec: [T: HasRegistry as r] => Codec[T] = r.registryCodec

given ToExpr[Identifier] = new ToExpr[Identifier]:
  override def apply(x: Identifier)(using q: Quotes): Expr[Identifier] =
    '{ Identifier.of(${Expr(x.getNamespace)}, ${Expr(x.getPath)}) }

private abstract class DeriverAnnotation extends MacroAnnotation:
  protected trait Context:
    val quotes: Quotes
    val classDef: quotes.reflect.ClassDef
    val companionDef: quotes.reflect.ClassDef

  protected def givenSymbol[T: Type](name: String)(using ctx: Context): ctx.quotes.reflect.Symbol =
    import ctx.quotes.reflect.*
    Symbol.newVal(ctx.companionDef.symbol, name, TypeRepr.of[T], Flags.Given | Flags.Synthetic, Symbol.noSymbol)

  protected def mkGiven[T: Type](using c: Context)(name: String, value: Expr[T]): c.quotes.reflect.ValDef =
    mkGiven(using c.quotes)(givenSymbol(name), value)
  protected def mkGiven[T](using q: Quotes)(name: q.reflect.Symbol, value: Expr[T]): q.reflect.ValDef =
    import q.reflect.*
    ValDef(name, Some(value.asTerm))

  def derive(using q: Quotes, ctx: Context { val quotes: q.type })(definition: q.reflect.ClassDef): Seq[q.reflect.Definition]
  override def transform(using q: Quotes)(definition: q.reflect.Definition, companion: Option[q.reflect.Definition]): List[q.reflect.Definition] =
    import q.reflect.*
    definition match
      case cl@ClassDef(name, _, _, _, _) =>
        companion match
          case Some(cd@ClassDef(cname, constr, parents, self, body)) =>
            given (Context { val quotes: q.type }) = new Context:
              override val quotes: q.type = q
              override val classDef: quotes.reflect.ClassDef = cl
              override val companionDef: quotes.reflect.ClassDef = cd
            return List(definition, ClassDef.copy(cd)(cname, constr, parents, self, body ++ derive(cl)))
          case None =>
            report.error(s"Please add a companion object to ${name}")
    List(definition) ++ companion

@deprecated
inline def summonUnlessSeeding[T]: T = summonInline[T]

// divert(-1)
/**
 *
*/
class hasCodec extends DeriverAnnotation:
  override def derive(using q: Quotes)(definition: q.reflect.ClassDef): Seq[q.reflect.Definition] =
    import q.reflect.*
    val flags: Flags = definition.symbol.flags
    if flags.is(Flags.Case) then
      if !flags.is(Flags.Final) then
        report.warning("Case class with codec should be final")

    else
      ???
// undivert