package org.eu.pool.fabric.inkling

import com.google.common.collect.ImmutableMultimap
import com.mojang.serialization.Codec
import io.github.apace100.origins.power.OriginsPowerTypes
import net.fabricmc.api.ModInitializer
import net.fabricmc.fabric.api.datagen.v1.FabricDataGenerator
import net.fabricmc.fabric.api.datagen.v1.FabricDataOutput
import net.fabricmc.fabric.api.datagen.v1.provider.FabricLanguageProvider
import net.fabricmc.fabric.api.datagen.v1.provider.FabricRecipeProvider
import net.fabricmc.fabric.api.datagen.v1.provider.FabricTagProvider
import net.minecraft.advancement.criterion.InventoryChangedCriterion
import net.minecraft.data.server.recipe.RecipeJsonProvider
import net.minecraft.data.server.recipe.ShapedRecipeJsonBuilder
import net.minecraft.entity.EquipmentSlot
import net.minecraft.entity.LivingEntity
import net.minecraft.entity.attribute.EntityAttribute
import net.minecraft.entity.attribute.EntityAttributeModifier
import net.minecraft.entity.attribute.EntityAttributes
import net.minecraft.entity.effect.StatusEffectInstance
import net.minecraft.entity.effect.StatusEffects
import net.minecraft.item.Item
import net.minecraft.item.ItemStack
import net.minecraft.item.Items
import net.minecraft.nbt.NbtCompound
import net.minecraft.nbt.NbtOps
import net.minecraft.recipe.book.RecipeCategory
import net.minecraft.registry.*
import net.minecraft.registry.entry.RegistryEntry
import net.minecraft.registry.tag.TagKey
import net.minecraft.server.world.ServerWorld
import net.minecraft.util.Identifier
import net.minecraft.util.Rarity
import net.minecraft.util.math.BlockPos
import net.minecraft.world.PersistentState
import org.eu.pool.fabric.inkling.PixelsInklingOrigin.applyCrimziumEffects
import org.eu.pool.fabric.inkling.mixin.SimpleRegistryAccessor
import org.slf4j.LoggerFactory
import java.util.*
import java.util.function.Consumer
import kotlin.jvm.optionals.getOrNull

const val modid = "inkling"

object CrimziumSword: Item(Settings().maxCount(1).fireproof().rarity(Rarity.RARE)) {
	override fun getAttributeModifiers(slot: EquipmentSlot?) = when (slot) {
		EquipmentSlot.MAINHAND -> ImmutableMultimap.builder<EntityAttribute, EntityAttributeModifier>().run {
			put(
				EntityAttributes.GENERIC_ATTACK_DAMAGE, EntityAttributeModifier(
					ATTACK_DAMAGE_MODIFIER_ID,
					"Weapon modifier",
					15.0,
					EntityAttributeModifier.Operation.ADDITION,
				)
			)
			put(
				EntityAttributes.GENERIC_ATTACK_SPEED, EntityAttributeModifier(
					ATTACK_SPEED_MODIFIER_ID,
					"Weapon modifier",
					-1.3,
					EntityAttributeModifier.Operation.ADDITION
				)
			)
			// TODO: figure out custom attack range / reach (+1 and +1.75)
			build()
		};
		else -> super.getAttributeModifiers(slot);
	}

	override fun postHit(stack: ItemStack?, target: LivingEntity?, attacker: LivingEntity?) =
		true.also { target?.applyCrimziumEffects() }
}

class WorldState(val pixelLastSavedPos: BlockPos?): PersistentState() {
	companion object {
		val CODEC = Codec.optionalField("pixelLastSavedPos", BlockPos.CODEC).codec().xmap({ WorldState(it.getOrNull()) }, { Optional.ofNullable(it.pixelLastSavedPos) })
		val ServerWorld.state get() = persistentStateManager.getOrCreate({ CODEC.decode(NbtOps.INSTANCE, it).getOrThrow(true) { throw IllegalStateException("Saved state for '$modid' corrupt: $it") }.first }, { WorldState(null) }, modid)
	}

	override fun writeNbt(nbt: NbtCompound?) = (nbt ?: NbtCompound()).also { CODEC.encode(this, NbtOps.INSTANCE, it) }
}

object CrimziumScythe: Item(Settings().maxCount(1).fireproof().rarity(Rarity.RARE)) {
	override fun getAttributeModifiers(slot: EquipmentSlot?) = when (slot) {
		EquipmentSlot.MAINHAND -> ImmutableMultimap.builder<EntityAttribute, EntityAttributeModifier>().run {
			put(
				EntityAttributes.GENERIC_ATTACK_DAMAGE, EntityAttributeModifier(
					ATTACK_DAMAGE_MODIFIER_ID,
					"Weapon modifier",
					17.5,
					EntityAttributeModifier.Operation.ADDITION,
				)
			)
			put(
				EntityAttributes.GENERIC_ATTACK_SPEED, EntityAttributeModifier(
					ATTACK_SPEED_MODIFIER_ID,
					"Weapon modifier",
					-1.0,
					EntityAttributeModifier.Operation.ADDITION
				)
			)
			// TODO: figure out custom reach (+1)
			build()
		};
		else -> super.getAttributeModifiers(slot);
	}

	override fun postHit(stack: ItemStack?, target: LivingEntity?, attacker: LivingEntity?) =
		true.also { target?.applyCrimziumEffects() }
}

object CrimziumIngot: Item(Settings().fireproof().rarity(Rarity.RARE))

object PixelsInklingOrigin : ModInitializer {
    private val logger = LoggerFactory.getLogger(modid)

	private val crimziumEffects get() = listOf(
		StatusEffectInstance(StatusEffects.DARKNESS, 5*20, 0),
		StatusEffectInstance(StatusEffects.WEAKNESS, 5*20, 0),
		StatusEffectInstance(StatusEffects.WITHER, 3*20, 0),
	)

	internal fun LivingEntity.applyCrimziumEffects() = crimziumEffects.forEach(this::addStatusEffect)

	override fun onInitialize() {
		// This code runs as soon as Minecraft is in a mod-load-ready state.
		// However, some things (like resources) may still be uninitialized.
		// Proceed with mild caution.
		logger.info("Hello Fabric world!")

		Registry.register(Registries.ITEM, "crimzium_sword".id, CrimziumSword)
		Registry.register(Registries.ITEM, "crimzium_scythe".id, CrimziumScythe)
		Registry.register(Registries.ITEM, "crimzium_ingot".id, CrimziumIngot)
		Registries.ITEM.rename(
			"roze_sword".id to CrimziumSword,
			"roze_scythe".id to CrimziumScythe,
			"roze_ingot".id to CrimziumIngot,
		)
	}

	fun datagen(gen: FabricDataGenerator) = gen.createPack().run {
		addProvider { out: FabricDataOutput ->
			object: FabricLanguageProvider(out) {
				override fun generateTranslations(b: TranslationBuilder) = b.run {
					add(CrimziumSword, "Crimzium Sword")
					add(CrimziumScythe, "Crimzium Scythe")
					add(CrimziumIngot, "Crimzium Ingot")
				}
			}
		}
		addProvider { out: FabricDataOutput, lookup ->
			object: FabricTagProvider.ItemTagProvider(out, lookup) {
				override fun configure(lookup: RegistryWrapper.WrapperLookup) {
					getOrCreateTagBuilder(RegistryKeys.ITEM.tagKey(".".id)).add(CrimziumSword, CrimziumScythe)
				}
			}
		}
		addProvider { out: FabricDataOutput ->
			object: FabricRecipeProvider(out) {
				override fun generate(c: Consumer<RecipeJsonProvider>) {
					ShapedRecipeJsonBuilder.create(RecipeCategory.COMBAT, CrimziumScythe)
						.pattern("rr")
						.pattern("/r")
						.pattern("/ ")
						.input('r', CrimziumIngot)
						.input('/', Items.STICK)
						.criterion("crimzium_ingot_for_scythe", InventoryChangedCriterion.Conditions.items(CrimziumIngot))
						.offerTo(c)
					ShapedRecipeJsonBuilder.create(RecipeCategory.COMBAT, CrimziumSword)
						.pattern("r")
						.pattern("r")
						.pattern("/ ")
						.input('r', CrimziumIngot)
						.input('/', Items.STICK)
						.criterion("crimzium_ingot_for_sword", InventoryChangedCriterion.Conditions.items(CrimziumIngot))
						.offerTo(c)
				}
			}
		}
	}

	inline fun <T> Registry<T>.rename(vararg aliases: Pair<Identifier, T>) = (this as SimpleRegistry).rename(*aliases)
	inline fun <T> SimpleRegistry<T>.rename(vararg aliases: Pair<Identifier, T>) = (this as SimpleRegistryAccessor<T>).run {
		aliases.forEach { (from, to) ->
			val idx = getKey(to).orElseThrow()
			keyToEntry[RegistryKey.of(key, from)] = keyToEntry[idx]
		}
	}

	val LivingEntity.doesNotSleep get() = false // TODO

	internal val String.id get() = Identifier(modid, this)
	internal fun <T> RegistryKey<Registry<T>>.tagKey(identifier: Identifier) = TagKey.of(this, identifier)
}
