@file:OptIn(ExperimentalUuidApi::class)
@file:Suppress("UnstableApiUsage")

package org.eu.net.pool.hexmu
import at.petrak.hexcasting.api.casting.*
import at.petrak.hexcasting.api.casting.castables.Action
import at.petrak.hexcasting.api.casting.castables.ConstMediaAction
import at.petrak.hexcasting.api.casting.castables.SpellAction
import at.petrak.hexcasting.api.casting.eval.CastResult
import at.petrak.hexcasting.api.casting.eval.CastingEnvironment
import at.petrak.hexcasting.api.casting.eval.ResolvedPatternType
import at.petrak.hexcasting.api.casting.eval.vm.CastingVM
import at.petrak.hexcasting.api.casting.eval.vm.SpellContinuation
import at.petrak.hexcasting.api.casting.iota.Iota
import at.petrak.hexcasting.api.casting.iota.IotaType
import at.petrak.hexcasting.api.casting.iota.NullIota
import at.petrak.hexcasting.api.casting.math.HexAngle
import at.petrak.hexcasting.api.casting.math.HexDir
import at.petrak.hexcasting.api.casting.math.HexPattern
import at.petrak.hexcasting.api.casting.mishaps.MishapBadCaster
import at.petrak.hexcasting.api.casting.mishaps.MishapInvalidIota
import at.petrak.hexcasting.api.item.IotaHolderItem
import at.petrak.hexcasting.api.misc.MediaConstants
import at.petrak.hexcasting.api.pigment.FrozenPigment
import at.petrak.hexcasting.api.utils.blue
import at.petrak.hexcasting.api.utils.putCompound
import at.petrak.hexcasting.common.blocks.akashic.BlockEntityAkashicBookshelf
import at.petrak.hexcasting.common.casting.actions.eval.OpEval
import at.petrak.hexcasting.common.lib.HexSounds
import dev.onyxstudios.cca.api.v3.component.Component
import dev.onyxstudios.cca.api.v3.component.ComponentKey
import dev.onyxstudios.cca.api.v3.component.ComponentRegistry
import dev.onyxstudios.cca.api.v3.component.sync.AutoSyncedComponent
import dev.onyxstudios.cca.api.v3.entity.EntityComponentFactoryRegistry
import dev.onyxstudios.cca.api.v3.entity.RespawnCopyStrategy
import dev.onyxstudios.cca.api.v3.world.WorldComponentFactoryRegistry
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidVariant
import net.minecraft.block.Block
import net.minecraft.block.BlockState
import net.minecraft.block.MapColor
import net.minecraft.entity.damage.DamageSource
import net.minecraft.entity.damage.DamageType
import net.minecraft.entity.player.PlayerEntity
import net.minecraft.item.Item
import net.minecraft.item.ItemStack
import net.minecraft.nbt.NbtCompound
import net.minecraft.nbt.NbtElement
import net.minecraft.registry.*
import net.minecraft.registry.entry.RegistryEntryList
import net.minecraft.server.MinecraftServer
import net.minecraft.server.world.ServerWorld
import net.minecraft.sound.SoundCategory
import net.minecraft.text.OrderedText
import net.minecraft.text.Style
import net.minecraft.text.Text
import net.minecraft.util.ActionResult
import net.minecraft.util.Formatting
import net.minecraft.util.Identifier
import net.minecraft.util.math.BlockPos
import net.minecraft.util.math.Vec3d
import net.minecraft.world.BlockView
import net.minecraft.world.World
import net.minecraft.world.explosion.Explosion
import net.minecraft.world.explosion.ExplosionBehavior
import net.minecraft.world.gen.chunk.FlatChunkGenerator
import net.minecraft.world.gen.chunk.FlatChunkGeneratorConfig
import net.minecraft.world.level.storage.LevelStorage
import org.eu.net.pool.common_curses.TextManipulator
import org.eu.net.pool.common_curses.compileJS
import org.mozilla.javascript.Context
import org.mozilla.javascript.Script
import org.mozilla.javascript.Scriptable
import org.mozilla.javascript.WrapFactory
import org.spongepowered.asm.mixin.injection.callback.CallbackInfoReturnable
import poollovernathan.fabric.execWithGlobals
import xyz.nucleoid.fantasy.Fantasy
import xyz.nucleoid.fantasy.RuntimeWorldConfig
import java.io.File
import java.io.Reader
import java.util.*
import java.util.concurrent.Executor
import kotlin.math.sin
import kotlin.uuid.ExperimentalUuidApi

const val modid = "hexmu"

val xplat = at.petrak.hexcasting.xplat.IXplatAbstractions.INSTANCE!!
val hexActions = xplat.actionRegistry

object Patterns {
    val createPocket = HexPatternBuilder.southeast.a.d.a.d.d.w.w.d.w.w.d.d.a.a.d.d.w()
    //region Hex metadata
    //region Inscriptions
    val inscribe = HexPatternBuilder.southwest.a.w.q.q.q.q.q.a.e.w.e.a.q.q.q.q.q.w.a()
    val inscription = HexPatternBuilder.northeast.e.e.e.e.e.a.a.q.e.w.e.q.a.a.e.e.e.e.e()
    val search = HexPatternBuilder.northwest.d.e.e.e.d.e.e.d()
    //endregion
    //region Executions
    val functor = HexPatternBuilder.east.d.e.e.e.e.e.q.q.a.e.d()
    val food = HexPatternBuilder.east.w.w.a.q.a.w.a.e.a.w.q.q.e.a.w.a()
    //endregion
    //region Annotations
    val annotate = HexPatternBuilder.northwest
        .q.q.e.d.e.e.e.e.e.a.a.q.e.e.d.a.q
        .q.a.d.e.e.q.a.a.e.e.e.e.e.d.e.q.q()
    val readAnnotation = HexPatternBuilder.northeast
        .e.e.e.e.e.a.a.q.e.e.d.a.q
        .q.a.d.e.e.q.a.a.e.e.e.e.e()
    //endregion
    //endregion
    //region Chat manipulation
    val uwuify = HexPatternBuilder.northeast.q.a.q.w.a.d.a.w.q.a.q()
    val getText = HexPatternBuilder.northwest
        .a.q.a.d.a.d.a.q.a()
    val getVision = HexPatternBuilder.northwest
        .q.q.q.q.q.d.a.d.q.q.q.q.q()
    val applyText = HexPatternBuilder.northeast
        .q.a.q.w.q.q.w.q.a.q.d.d.a.a.d.w.d.a.a.d
        .d.w.e.w.w.w.e.d.a.a.d.d.w.a.d.q.d.q.d()
    val silence = HexPatternBuilder.east.w.a.q.a.a()
    val unsilence = HexPatternBuilder.east.w.a.q.a.e()
    //endregion
    //region Pockets
    val conjurePocket = HexPatternBuilder.east
        .q.w.q.w.q.w.q.w.q.w.q.a.q.d
        .q.d.q.d.q.d.q.d.d.a.d.a.a.d
        .a.a.d.e.w.e.e.e.a.e.e.e.a.e
        .e.e.a.e.e.e.a.e.e.e.a.e.e.e()
    val dismissPocket = HexPatternBuilder.east
        .w.w.w.d.e.d.w.w.w.a.w.q.w.a.q
        .a.w.q.w.q.q.d.q.q.a.a.q.a.e.w
        .w.d.e.d.w.w.w.d.d.e.d.a.q.d.q()
    //endregion
    //region Boxes
    val boxIota = HexPatternBuilder.southeast.a.a.d.d()
    val unboxIota = HexPatternBuilder.northeast.d.d.a.a()
    val boxPipebomb = HexPatternBuilder.southeast.a.a.d.d.q.a.a.q.a()
    //endregion
}

//region Text manipulation

//endregion

object SyntheticFluidItem: Item(Settings()) {
    fun getFluid(stack: ItemStack): FluidVariant = FluidVariant.fromNbt(stack.nbt)
    fun setFluid(stack: ItemStack, fluid: FluidVariant) { stack.nbt = fluid.toNbt() }
    override fun getName(): Text = Text.literal("You should never see this")
    override fun getName(stack: ItemStack) = getFluid(stack).fluid.defaultState?.blockState?.block?.name ?: Text.literal("Fluid")!!
}

object FillerBlock: Block(Settings.create().mapColor(MapColor.CLEAR).replaceable().dropsNothing().hardness(0.2f))

object Actions {
    val uwuify = object: SpellAction {
        override val argc = 1

        override fun execute(
            args: List<Iota>,
            env: CastingEnvironment
        ): SpellAction.Result {
            val player = args.getPlayer(0, argc)
            val component = player.getComponent(playerStateComponent)
            return SpellAction.Result(object: RenderedSpell {
                override fun cast(env: CastingEnvironment) {
                    component.chatManip = TextEffect.UWU
                    playerStateComponent.sync(player)
                }
            }, if (component.chatManip == TextEffect.UWU) 0 else MediaConstants.CRYSTAL_UNIT * 3, listOf())
        }
    }

    const val INSCRIPTION_KEY = "$modid:inscription"
    const val FUNCTOR_KEY = "$modid:functor"
    const val EXPIRY_KEY = "$modid:expiry"
    const val ANNOTATION_KEY = "$modid:annotations"

    val inscribe = object: ConstMediaAction {
        override val argc = 1

        override fun execute(
            args: List<Iota>,
            env: CastingEnvironment
        ): List<Iota> {
            val (arg) = args
            check(arg is IotaDuck)
            return listOf(arg.inscription ?: NullIota())
        }
    }

    val inscription = object: ConstMediaAction {
        override val argc = 2

        override fun execute(
            args: List<Iota>,
            env: CastingEnvironment
        ): List<Iota> {
            val (target, label) = args
            check(target is IotaDuck)
            println("Planning to inscribe $target with $label")
            // This feels evil.
            val data = IotaType.serialize(target)
            println("Serialized data: $data")
            if (label is NullIota) {
                data.remove(INSCRIPTION_KEY)
                println("Removed inscription")
            } else {
                data.put(INSCRIPTION_KEY, IotaType.serialize(label))
                println("Added inscription")
            }
            println("New data: $data")
            val new = IotaType.deserializeIota(data, env.world)
            return listOf(new)
        }
    }

    val search = object: ConstMediaAction {
        override val argc = 2

        override fun execute(
            args: List<Iota>,
            env: CastingEnvironment
        ): List<Iota> = listOf(args.getList(0, argc).find { it is IotaDuck && it.inscription != null && Iota.tolerates(it.inscription, args[1]) }.orNull())
    }

    val functor = object: ConstMediaAction {
        override val argc = 2

        override fun execute(
            args: List<Iota>,
            env: CastingEnvironment
        ): List<Iota> {
            val (target, label) = args
            check(target is IotaDuck)
            // This feels evil x2
            val data = IotaType.serialize(target)
            if (label is NullIota) {
                data.remove(FUNCTOR_KEY)
            } else {
                data.put(FUNCTOR_KEY, IotaType.serialize(label))
            }
            return listOf(IotaType.deserializeIota(data, env.world))
        }
    }

    val silence = object: SpellAction {
        override val argc = 1

        override fun execute(args: List<Iota>, env: CastingEnvironment): SpellAction.Result {
            val player = args.getPlayer(0, argc)
            val state = player.getComponent(playerStateComponent)
            return SpellAction.Result(object: RenderedSpell {
                override fun cast(env: CastingEnvironment) {
                    state.textManip = TextEffect.EMPTY
                    playerStateComponent.sync(player)
                }
            }, if (state.textManip == TextEffect.EMPTY) 0 else MediaConstants.QUENCHED_BLOCK_UNIT, listOf())
        }
    }

    val unsilence = object: SpellAction {
        override val argc = 1

        override fun execute(args: List<Iota>, env: CastingEnvironment): SpellAction.Result {
            val player = args.getPlayer(0, argc)
            val state = player.getComponent(playerStateComponent)
            return SpellAction.Result(object: RenderedSpell {
                override fun cast(env: CastingEnvironment) {
                    state.textManip = TextEffect.NONE
                    playerStateComponent.sync(player)
                }
            }, if (state.textManip == TextEffect.NONE) 0 else MediaConstants.QUENCHED_BLOCK_UNIT * 12, listOf())
        }
    }

    val createPocket = object: ConstMediaAction {
        override val argc = 0
        override val mediaCost = 8 * MediaConstants.QUENCHED_BLOCK_UNIT
        override fun execute(
            args: List<Iota>,
            env: CastingEnvironment
        ): List<Iota> {
            val name = Identifier(modid, "pocket/${UUID.randomUUID()}")
            val key = RegistryKey.of(RegistryKeys.WORLD, name)
            // this works for now
            val iota = PocketDimIota(key, env.world.server, env.world.registryManager, PocketDimInfo(pigment = env.pigment, worldSeed = env.world.random.nextLong()))
            print("mlem $iota")
            return listOf(iota)
        }
    }

    val box = object: ConstMediaAction {
        override val argc = 1
        override fun execute(
            args: List<Iota>,
            env: CastingEnvironment
        ) = listOf(BoxIota(IotaType.serialize(args[0])))
    }

    val unbox = object: ConstMediaAction {
        override val argc = 1
        override fun execute(
            args: List<Iota>,
            env: CastingEnvironment
        ): List<Iota> {
            val (box) = args
            if (box !is BoxIota) {
                throw MishapInvalidIota(box, 0, BoxIota.typeName())
            }
            if (box.data.getBoolean("pipebomb")) {
                env.world.createExplosion(null, DamageSource(env.world.registryManager[RegistryKeys.DAMAGE_TYPE].getEntry(pipeBombDamageTypeKey).orElseThrow()), object: ExplosionBehavior() {
                    override fun canDestroyBlock(
                        explosion: Explosion?,
                        world: BlockView?,
                        pos: BlockPos?,
                        state: BlockState?,
                        power: Float
                    ) = false
                }, env.castingEntity?.pos ?: throw MishapBadCaster(), 3.0f, false, World.ExplosionSourceType.TNT)
            }
            return listOf(IotaType.deserializeIota(box.data, env.world))
        }
    }
}

data class BoxIota(val data: NbtCompound): Iota(BoxIota, data) {
    override fun isTruthy() = true
    override fun toleratesOther(p0: Iota) = p0 is BoxIota && data == p0.data
    override fun serialize() = data
    companion object: IotaType<BoxIota>() {
        override fun deserialize(
            p0: NbtElement,
            p1: ServerWorld?
        ) = BoxIota(p0 as? NbtCompound ?: NbtCompound())

        override fun display(p0: NbtElement?) = Text.translatable("Box").blue
        override fun color() = Formatting.BLUE.colorValue!!

        val pipeBomb = BoxIota(NbtCompound().also { it.putBoolean("pipebomb", true) })
    }
}

@Suppress("INAPPLICABLE_JVM_NAME")
interface MinecraftServerDuck {
    val executor: Executor
        @JvmName("hexmu\$getExecutor") get
    val session: LevelStorage.Session
        @JvmName("hexmu\$getSession") get
    var worlds: Map<RegistryKey<World>, ServerWorld>
        @JvmName("hexmu\$getWorlds") get
        @JvmName("hexmu\$setWorlds") set
}

val pipeBombDamageTypeKey: RegistryKey<DamageType> = RegistryKey.of(RegistryKeys.DAMAGE_TYPE, id("pipe_bomb"))

private val configDelegate = lazy(Object()) {
    class SpellManager(val default: Boolean = true, vararg spells: String) {
        val enabled = mutableMapOf<String, Boolean>()
        init {
            spells.forEach {
                enabled[it] = default
            }
        }
        fun keys() = enabled.keys.toTypedArray()
        fun set(name: String, value: Boolean) {
            require(enabled.contains(name)) { "Attempt to modify nonexistent spell $name." }
            enabled[name] = value
        }
        fun all(value: Boolean) {
            enabled.keys.forEach { enabled[it] = value }
        }
        operator fun get(key: String) = checkNotNull(enabled[key])
    }
    class Spells {
        @JvmField val chatManipulation = SpellManager(true, "uwuify", "silence", "unsilence")
        @JvmField val boxes = SpellManager(true, "box", "unbox", "pipebomb")
        @JvmField val inscriptions = SpellManager(true, "getInscription", "setInscription", "setFunctor", "lookupInscription", "getAnnotation", "setAnnotation")
        @JvmField val pockets = SpellManager(true, "create")
        @JvmField val all = arrayOf(chatManipulation, boxes, inscriptions, pockets)
    }
    class Config {
        @JvmField val spells = Spells()
    }
    val config = Config()
    val path = File("kubejs/hexmu.js")
    if (path.exists()) {
        compileJS(path.reader(Charsets.UTF_8)).execWithGlobals(mapOf("config" to config))
    }
    config
}
private val config by configDelegate

data class PocketDimIota(val key: RegistryKey<World>, val server: MinecraftServer, val manager: DynamicRegistryManager, val info: PocketDimInfo): Iota(PocketDimIota, key to info) {
    override fun isTruthy() = manager[RegistryKeys.WORLD].contains(key)
    override fun toleratesOther(p0: Iota) = p0 is PocketDimIota && key.value == p0.key.value
    override fun serialize() = NbtCompound().apply {
        putString("key", key.value.toString())
        putCompound("info", info.toNBT())
    }

    val pocketConfig = RuntimeWorldConfig().apply {
        generator = FlatChunkGenerator(FlatChunkGeneratorConfig(
            Optional.of(RegistryEntryList.of()),
            manager[RegistryKeys.BIOME].getEntry(
                RegistryKey.of(RegistryKeys.BIOME, Identifier("minecraft", "the_void"))
            ).orElseThrow(),
            listOf(),
        ))
        seed = info.worldSeed
    }

    val handle = Fantasy.get(server).getOrOpenPersistentWorld(key.value, pocketConfig)!!

    init {
        pocketDims.putIfAbsent(key.value.toString(), info)
    }

    companion object: IotaType<PocketDimIota>() {
        val pocketDims = mutableMapOf<String, PocketDimInfo>()
        override fun deserialize(
            p0: NbtElement,
            p1: ServerWorld
        ): PocketDimIota? {
            if (p0 !is NbtCompound) return null
            val key = Identifier.tryParse(p0.getString("key")) ?: return null
            val info = PocketDimInfo()
            info.fromNBT(p0.getCompound("info"))
            return PocketDimIota(
                RegistryKey.of(RegistryKeys.WORLD, key),
                p1.server,
                p1.registryManager,
                info,
            )
        }

        val timeValue = sin(System.currentTimeMillis() / 10000.0).toFloat()

        override fun display(p0: NbtElement): Text {
            if (p0 is NbtCompound) {
                val key = p0.getString("key").takeLast(6)
                val text = Text.translatable("[Pocket %s]", Text.literal(key).styled { it.withFont(Identifier("minecraft", "illageralt")) })
                val pigment = p0.getCompound("info").getCompound("pigment")
                if (!pigment.isEmpty) text.styled { it.withColor(FrozenPigment.fromNBT(pigment).colorProvider.getColor(timeValue, Vec3d.ZERO)) }
                return text
            }
            return Text.literal("[Pocket]")
        }

        override fun color() = FrozenPigment.DEFAULT.get().colorProvider.getColor(timeValue, Vec3d.ZERO)
    }
}

object MediaStrand: Item(Settings()), IotaHolderItem {
    override fun readIotaTag(p0: ItemStack) = p0.nbt!!.getCompound("iota")!!
    override fun writeable(p0: ItemStack) = false
    override fun canWrite(p0: ItemStack, p1: Iota?) = false
    override fun writeDatum(p0: ItemStack, p1: Iota?) {}
    fun of(iota: Iota) = ItemStack(MediaStrand, 1).also {
        it.orCreateNbt.putCompound("iota", IotaType.serialize(iota))
    }
}

data class PocketDimInfo(var pigment: FrozenPigment = FrozenPigment.DEFAULT.get(), var worldSeed: Long = 0L) {
    fun fromNBT(p0: NbtCompound) {
        if (p0.contains("pigment", NbtElement.COMPOUND_TYPE.toInt())) {
            pigment = FrozenPigment.fromNBT(p0.getCompound("pigment"))
        } else {
            pigment = FrozenPigment.DEFAULT.get()
        }
    }

    fun toNBT() = NbtCompound().apply {
        putCompound("pigment", pigment.serializeToNBT())
    }
}

data class PocketDimMeta(val world: World, val info: PocketDimInfo = PocketDimInfo()): Component, AutoSyncedComponent {
    override fun readFromNbt(p0: NbtCompound) {
        info.fromNBT(p0)
    }

    override fun writeToNbt(p0: NbtCompound) {
        p0.copyFrom(info.toNBT())
    }
}

enum class TextEffect(val pattern: HexPattern, val manip: TextManipulator) {
    NONE(Patterns.unsilence, TextManipulator.None),
    UWU(Patterns.uwuify, TextManipulator.UwU),
    EMPTY(Patterns.silence, TextManipulator.Empty),
}

@Suppress("INAPPLICABLE_JVM_NAME")
interface IotaDuck {
    var inscription: Iota?
        @JvmName("hexmu\$getInscription") get
        @JvmName("hexmu\$setInscription") set
    var functor: Iota?
        @JvmName("hexmu\$getFunctor") get
        @JvmName("hexmu\$setFunctor") set
    var expiry: Long
        @JvmName("hexmu\$getExpiry") get
        @JvmName("hexmu\$setExpiry") set
    var annotations: MutableMap<Iota, Iota>
        @JvmName("hexmu\$getAnnotations") get
        @JvmName("hexmu\$setAnnotations") set
    var server: MinecraftServer
        @JvmName("hexmu\$getServer") get
        @JvmName("hexmu\$setServer") set
}

fun id(value: String) = Identifier(modid, value)
data class PlayerState
@JvmOverloads constructor(
    val player: PlayerEntity,
    var chatManip: TextEffect = TextEffect.NONE,
    var textManip: TextEffect = TextEffect.NONE,
): Component, AutoSyncedComponent {
    override fun readFromNbt(p0: NbtCompound) {
        if (p0.getString("chat") != "") {
            chatManip = TextEffect.valueOf(p0.getString("chat"))
        } else {
            // compatibility
            if (p0.getBoolean("uwuify")) {
                chatManip = TextEffect.UWU
            }
            if (p0.getBoolean("silence")) {
                chatManip = TextEffect.EMPTY
            }
        }
        if (p0.getString("text") != "") {
            chatManip = TextEffect.valueOf(p0.getString("text"))
        }
    }

    override fun writeToNbt(p0: NbtCompound) {
        p0.putString("chat", chatManip.name)
        p0.putString("text", textManip.name)
    }
}

val playerStateComponent: ComponentKey<PlayerState> by lazy {
    ComponentRegistry.getOrCreate(id("player_state"), PlayerState::class.java)
}
val pocketDataComponent: ComponentKey<PocketDimMeta> by lazy {
    ComponentRegistry.getOrCreate(id("pocket_data"), PocketDimMeta::class.java)
}

@Deprecated("fucking sucks", level = DeprecationLevel.ERROR)
fun init() {
    Registry.register(Registries.BLOCK, id("filler"), FillerBlock)
    Registry.register(Registries.ITEM, id("iota"), MediaStrand)
    if (config.spells.chatManipulation["uwuify"]) Registry.register(hexActions, id("uwuify"), ActionRegistryEntry(Patterns.uwuify, Actions.uwuify))
    if (config.spells.inscriptions["getInscription"]) Registry.register(hexActions, id("inscribe"), ActionRegistryEntry(Patterns.inscribe, Actions.inscribe))
    if (config.spells.inscriptions["setInscription"]) Registry.register(hexActions, id("inscription"), ActionRegistryEntry(Patterns.inscription, Actions.inscription))
    if (config.spells.inscriptions["lookupInscription"]) Registry.register(hexActions, id("substring"), ActionRegistryEntry(Patterns.search, Actions.search))
    if (config.spells.inscriptions["setFunctor"]) Registry.register(hexActions, id("functor"), ActionRegistryEntry(Patterns.functor, Actions.functor))
    if (config.spells.pockets["create"]) Registry.register(hexActions, id("pocket/open"), ActionRegistryEntry(Patterns.conjurePocket, Actions.createPocket))
    //Registry.register(hexActions, id("pocket/close"), ActionRegistryEntry(Patterns.dismissPocket, Actions.functor))
    if (config.spells.chatManipulation["silence"]) Registry.register(hexActions, id("silence"), ActionRegistryEntry(Patterns.silence, Actions.silence))
    if (config.spells.chatManipulation["unsilence"]) Registry.register(hexActions, id("unsilence"), ActionRegistryEntry(Patterns.unsilence, Actions.unsilence))
    if (config.spells.boxes["box"]) Registry.register(hexActions, id("box"), ActionRegistryEntry(Patterns.boxIota, Actions.box))
    if (config.spells.boxes["unbox"]) Registry.register(hexActions, id("unbox"), ActionRegistryEntry(Patterns.unboxIota, Actions.unbox))
    if (config.spells.boxes["pipebomb"]) Registry.register(hexActions, id("pipebomb"), ActionRegistryEntry(Patterns.boxPipebomb, Action.makeConstantOp(BoxIota.pipeBomb)))
    if (config.spells.inscriptions["setAnnotation"]) Registry.register(hexActions, id("annotation/get"), ActionRegistryEntry(Patterns.annotate, object: ConstMediaAction {
        override val argc = 3
        override fun execute(args: List<Iota>, env: CastingEnvironment): List<Iota> {
            val (target, key, value) = args
            val copy = IotaType.deserializeIota(IotaType.serialize(target), env.world)
            check(copy is IotaDuck)
            println("annotating $target (copied to $copy) with $key → $value")
            val realKey = copy.annotations.keys.find { Iota.tolerates(it, key) } ?: key
            if (value is NullIota) {
                copy.annotations.remove(realKey)
            } else {
                copy.annotations[key] = value
            }
            return listOf(copy)
        }
    }))
    if (config.spells.inscriptions["getAnnotation"]) Registry.register(hexActions, id("annotation/set"), ActionRegistryEntry(Patterns.readAnnotation, object: ConstMediaAction {
        override val argc = 2
        override fun execute(args: List<Iota>, env: CastingEnvironment) = listOf((args[0] as? IotaDuck)?.annotations?.let { it[args[1]] ?: it.entries.find { Iota.tolerates(args[1], it.key) }?.value }.orNull())
    }))
    Registry.register(xplat.iotaTypeRegistry, id("pocket"), PocketDimIota)
    Registry.register(xplat.iotaTypeRegistry, id("box"), BoxIota)
}

val Iota.duck get() = this as IotaDuck

//object ConfigManager {
//    lateinit var configScript
//    lateinit var configObject
//    fun reload() {
//        configScript = compileJS()
//    }
//}

fun EntityComponentFactoryRegistry.initComponents() {
    registerForPlayers(playerStateComponent, ::PlayerState, RespawnCopyStrategy.CHARACTER)
}

fun WorldComponentFactoryRegistry.initWorldComponents() {
    register(pocketDataComponent, ::PocketDimMeta)
}

fun invokeWithFunctor(
    vm: CastingVM,
    world: ServerWorld,
    cont: SpellContinuation,
    functor: Iota,
): CastResult {
    val (image, effects, newCont, sound) = OpEval.exec(vm.env, vm.image, cont, vm.image.stack.toMutableList(), functor)
    return CastResult(
        functor,
        newCont,
        image,
        effects,
        ResolvedPatternType.EVALUATED,
        sound,
    )
}

class HexPatternBuilder(val startDir: HexDir) {
    var angles = mutableListOf<HexAngle>()

    companion object {
        val northeast get() = HexPatternBuilder(HexDir.NORTH_EAST)
        val east get() = HexPatternBuilder(HexDir.EAST)
        val southeast get() = HexPatternBuilder(HexDir.SOUTH_EAST)
        val southwest get() = HexPatternBuilder(HexDir.SOUTH_WEST)
        val west get() = HexPatternBuilder(HexDir.WEST)
        val northwest get() = HexPatternBuilder(HexDir.NORTH_WEST)
    }

    val a get() = also { angles += HexAngle.LEFT_BACK }
    val q get() = also { angles += HexAngle.LEFT }
    val w get() = also { angles += HexAngle.FORWARD }
    val e get() = also { angles += HexAngle.RIGHT }
    val d get() = also { angles += HexAngle.RIGHT_BACK }
    val s get() = also { angles += HexAngle.BACK }

    operator fun invoke() = HexPattern(startDir, angles.toMutableList())
}

@Suppress("INAPPLICABLE_JVM_NAME")
interface RawIotaHolderItem: IotaHolderItem {
    var ItemStack.iotaTag: NbtCompound?
        @JvmName("readIotaTag") @Suppress("ACCIDENTAL_OVERRIDE") get
        @JvmName("hexmu\$writeIotaTag") set
}

fun copyFromBookself(
    shelf: BlockEntityAkashicBookshelf,
    world: World,
    pos: BlockPos,
    player: PlayerEntity,
    stack: ItemStack,
    cir: CallbackInfoReturnable<ActionResult>
) {
    val item = stack.item
    if (item is IotaHolderItem) {
        if (shelf.iotaTag == null || !item.writeable(stack)) {
            cir.returnValue = ActionResult.FAIL
            return
        }
        if (item is RawIotaHolderItem) {
            if (!world.isClient) {
                with(item) {
                    stack.iotaTag = shelf.iotaTag
                }
            }
            world.playSound(player, pos, HexSounds.SCROLL_SCRIBBLE, SoundCategory.BLOCKS, 1f, 1f);
            cir.returnValue = ActionResult.SUCCESS
            return
        }
        // we can only hunch
        if (world is ServerWorld) {
            val iota = IotaType.deserializeIota(shelf.iotaTag, world)
            if (item.canWrite(stack, iota)) {
                item.writeDatum(stack, iota);
                world.playSound(null, pos, HexSounds.SCROLL_SCRIBBLE, SoundCategory.BLOCKS, 1f, 1f);
            }
        }
        cir.returnValue = ActionResult.SUCCESS
    }
}
