@file:OptIn(ExperimentalUuidApi::class)
@file:Suppress("UnstableApiUsage")

package org.eu.net.pool.hexmu
import at.petrak.hexcasting.api.casting.*
import at.petrak.hexcasting.api.casting.castables.Action
import at.petrak.hexcasting.api.casting.castables.ConstMediaAction
import at.petrak.hexcasting.api.casting.castables.SpellAction
import at.petrak.hexcasting.api.casting.eval.CastResult
import at.petrak.hexcasting.api.casting.eval.CastingEnvironment
import at.petrak.hexcasting.api.casting.eval.ResolvedPatternType
import at.petrak.hexcasting.api.casting.eval.vm.CastingVM
import at.petrak.hexcasting.api.casting.eval.vm.SpellContinuation
import at.petrak.hexcasting.api.casting.iota.Iota
import at.petrak.hexcasting.api.casting.iota.IotaType
import at.petrak.hexcasting.api.casting.iota.NullIota
import at.petrak.hexcasting.api.casting.math.HexAngle
import at.petrak.hexcasting.api.casting.math.HexDir
import at.petrak.hexcasting.api.casting.math.HexPattern
import at.petrak.hexcasting.api.casting.mishaps.MishapBadCaster
import at.petrak.hexcasting.api.casting.mishaps.MishapInvalidIota
import at.petrak.hexcasting.api.item.IotaHolderItem
import at.petrak.hexcasting.api.misc.MediaConstants
import at.petrak.hexcasting.api.pigment.FrozenPigment
import at.petrak.hexcasting.api.utils.blue
import at.petrak.hexcasting.api.utils.getUUID
import at.petrak.hexcasting.api.utils.putCompound
import at.petrak.hexcasting.common.blocks.akashic.BlockEntityAkashicBookshelf
import at.petrak.hexcasting.common.casting.actions.eval.OpEval
import at.petrak.hexcasting.common.lib.HexSounds
import dev.onyxstudios.cca.api.v3.component.Component
import dev.onyxstudios.cca.api.v3.component.ComponentKey
import dev.onyxstudios.cca.api.v3.component.ComponentRegistry
import dev.onyxstudios.cca.api.v3.component.sync.AutoSyncedComponent
import dev.onyxstudios.cca.api.v3.entity.EntityComponentFactoryRegistry
import dev.onyxstudios.cca.api.v3.entity.RespawnCopyStrategy
import dev.onyxstudios.cca.api.v3.world.WorldComponentFactoryRegistry
import kotlinx.datetime.Instant
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidVariant
import net.minecraft.block.Block
import net.minecraft.block.BlockState
import net.minecraft.block.MapColor
import net.minecraft.entity.damage.DamageSource
import net.minecraft.entity.damage.DamageType
import net.minecraft.entity.player.PlayerEntity
import net.minecraft.item.Item
import net.minecraft.item.ItemStack
import net.minecraft.nbt.NbtCompound
import net.minecraft.nbt.NbtElement
import net.minecraft.registry.*
import net.minecraft.server.MinecraftServer
import net.minecraft.server.WorldGenerationProgressListener
import net.minecraft.server.world.ServerWorld
import net.minecraft.sound.SoundCategory
import net.minecraft.text.Text
import net.minecraft.util.ActionResult
import net.minecraft.util.Formatting
import net.minecraft.util.Identifier
import net.minecraft.util.math.BlockPos
import net.minecraft.util.math.ChunkPos
import net.minecraft.util.math.Vec3d
import net.minecraft.world.BlockView
import net.minecraft.world.World
import net.minecraft.world.border.WorldBorderListener
import net.minecraft.world.chunk.ChunkStatus
import net.minecraft.world.explosion.Explosion
import net.minecraft.world.explosion.ExplosionBehavior
import net.minecraft.world.level.UnmodifiableLevelProperties
import net.minecraft.world.level.storage.LevelStorage
import org.mozilla.javascript.Context
import org.mozilla.javascript.Script
import org.mozilla.javascript.Scriptable
import org.mozilla.javascript.WrapFactory
import org.spongepowered.asm.mixin.injection.callback.CallbackInfoReturnable
import software.bernie.shadowed.eliotlash.mclib.math.functions.limit.Min
import java.io.Reader
import java.util.*
import java.util.concurrent.Executor
import kotlin.math.sin
import kotlin.uuid.ExperimentalUuidApi
import kotlin.uuid.Uuid
import kotlin.uuid.toJavaUuid
import kotlin.uuid.toKotlinUuid

const val modid = "hexmu"

val xplat = at.petrak.hexcasting.xplat.IXplatAbstractions.INSTANCE!!
val hexActions = xplat.actionRegistry

object Patterns {
    val createPocket = HexPatternBuilder.southeast.a.d.a.d.d.w.w.d.w.w.d.d.a.a.d.d.w()
    //region Hex metadata
    //region Inscriptions
    val inscribe = HexPatternBuilder.southwest.a.w.q.q.q.q.q.a.e.w.e.a.q.q.q.q.q.w.a()
    val inscription = HexPatternBuilder.northeast.e.e.e.e.e.a.a.q.e.w.e.q.a.a.e.e.e.e.e()
    val search = HexPatternBuilder.northwest.d.e.e.e.d.e.e.d()
    //endregion
    //region Executions
    val functor = HexPatternBuilder.east.d.e.e.e.e.e.q.q.a.e.d()
    val food = HexPatternBuilder.east.w.w.a.q.a.w.a.e.a.w.q.q.e.a.w.a()
    //endregion
    //region Annotations
    val annotate = HexPatternBuilder.northwest
        .q.q.e.d.e.e.e.e.e.a.a.q.e.e.d.a.q
        .q.a.d.e.e.q.a.a.e.e.e.e.e.d.e.q.q()
    val readAnnotation = HexPatternBuilder.northeast
        .e.e.e.e.e.a.a.q.e.e.d.a.q
        .q.a.d.e.e.q.a.a.e.e.e.e.e()
    //endregion
    //endregion
    //region Chat manipulation
    val uwuify = HexPatternBuilder.northeast.q.a.q.w.a.d.a.w.q.a.q()
    val silence = HexPatternBuilder.east.w.a.q.a.a()
    val unsilence = HexPatternBuilder.east.w.a.q.a.e()
    //endregion
    //region Pockets
    val conjurePocket = HexPatternBuilder.east
        .q.w.q.w.q.w.q.w.q.w.q.a.q.d
        .q.d.q.d.q.d.q.d.d.a.d.a.a.d
        .a.a.d.e.w.e.e.e.a.e.e.e.a.e
        .e.e.a.e.e.e.a.e.e.e.a.e.e.e()
    val dismissPocket = HexPatternBuilder.east
        .w.w.w.d.e.d.w.w.w.a.w.q.w.a.q
        .a.w.q.w.q.q.d.q.q.a.a.q.a.e.w
        .w.d.e.d.w.w.w.d.d.e.d.a.q.d.q()
    //endregion
    //region Boxes
    val boxIota = HexPatternBuilder.northeast.a.a.d.d()
    val unboxIota = HexPatternBuilder.northeast.d.d.a.a()
    val boxPipebomb = HexPatternBuilder.southeast.a.a.d.d.q.a.a.q.a()
    //endregion
}

//region Text manipulation
private val UWUFY_PHRASES = arrayOf(
    "UwU",
    "owo",
    "OwO",
    "uwu",
    ">w<",
    "^w^",
    ":3",
    "^-^",
    "^_^",
    "^w^",
    ":3"
)

// Copied from https://github.com/MayaqqDev/Cynosure/blob/master/common/src/main/kotlin/dev/mayaqq/cynosure/utils/fun/UwUfy.kt
// Maya uses Mojmap, so a proper dependency is not an option
fun String.uwufy(): String {
    var input = this
    val stringLength = input.length
    // Replace 'r' and 'l' with 'w', and 'R' and 'L' with 'W'
    // Replace 'ove' with 'uv' and 'OVE' with 'UV'
    // Replace 'o' with 'owo' and 'O' with 'OwO'
    // Replace repeated exclamation marks and question marks
    input = input
        .replace("[rl]".toRegex(), "w").replace("[RL]".toRegex(), "W")
        .replace("ove".toRegex(), "uv").replace("OVE".toRegex(), "UV")
        .replace("o".toRegex(), "owo").replace("O".toRegex(), "OwO")
        .replace("!".toRegex(), "!!!").replace("\\?".toRegex(), "???")

    // Convert to uppercase
    if (stringLength % 3 == 0) {
        input = input.uppercase()
    }

    input = input.replace(Regex("%(\\p{L})")) { m -> "%" + m.groupValues[1].lowercase() }
    input = input.replace(Regex("\\$(\\p{L})")) { m -> "\\$" + m.groupValues[1].lowercase() }

    input = if (stringLength % 2 == 0) {
        // Add more letters to the end of words (Not numbers!)
        input.replace("(\\p{L})(\\b)".toRegex(), "$1$1$1$1$2")
    } else {
        // 50% chance to duplicate the first letter and add '-'
        input.replace("\\b(\\p{L})(\\p{L}*)\\b".toRegex(), "$1-$1$2")
    }

    return input + " " + UWUFY_PHRASES[stringLength % UWUFY_PHRASES.size]
}
//endregion

object SyntheticFluidItem: Item(Settings()) {
    fun getFluid(stack: ItemStack): FluidVariant = FluidVariant.fromNbt(stack.nbt)
    fun setFluid(stack: ItemStack, fluid: FluidVariant) { stack.nbt = fluid.toNbt() }
    override fun getName(): Text = Text.literal("You should never see this")
    override fun getName(stack: ItemStack) = getFluid(stack).fluid.defaultState?.blockState?.block?.name ?: Text.literal("Fluid")!!
}

object FillerBlock: Block(Settings.create().mapColor(MapColor.CLEAR).replaceable().dropsNothing().hardness(0.2f))

object Actions {
    val uwuify = object: SpellAction {
        override val argc = 1

        override fun execute(
            args: List<Iota>,
            env: CastingEnvironment
        ): SpellAction.Result {
            val player = args.getPlayer(0, argc)
            val component = player.getComponent(playerStateComponent)
            return SpellAction.Result(object: RenderedSpell {
                override fun cast(env: CastingEnvironment) {
                    component.uwuify = true
                    component.uwuifyCause = env.castingEntity?.uuid?.toKotlinUuid() ?: Uuid.NIL
                    playerStateComponent.sync(player)
                }
            }, if (component.uwuify) 0 else MediaConstants.CRYSTAL_UNIT * 3, listOf())
        }
    }

    const val INSCRIPTION_KEY = "$modid:inscription"
    const val FUNCTOR_KEY = "$modid:functor"
    const val EXPIRY_KEY = "$modid:expiry"
    const val ANNOTATION_KEY = "$modid:annotations"

    val inscribe = object: ConstMediaAction {
        override val argc = 1

        override fun execute(
            args: List<Iota>,
            env: CastingEnvironment
        ): List<Iota> {
            val (arg) = args
            check(arg is IotaDuck)
            return listOf(arg.inscription ?: NullIota())
        }
    }

    val inscription = object: ConstMediaAction {
        override val argc = 2

        override fun execute(
            args: List<Iota>,
            env: CastingEnvironment
        ): List<Iota> {
            val (target, label) = args
            check(target is IotaDuck)
            println("Planning to inscribe $target with $label")
            // This feels evil.
            val data = IotaType.serialize(target)
            println("Serialized data: $data")
            if (label is NullIota) {
                data.remove(INSCRIPTION_KEY)
                println("Removed inscription")
            } else {
                data.put(INSCRIPTION_KEY, IotaType.serialize(label))
                println("Added inscription")
            }
            println("New data: $data")
            val new = IotaType.deserializeIota(data, env.world)
            return listOf(new)
        }
    }

    val search = object: ConstMediaAction {
        override val argc = 2

        override fun execute(
            args: List<Iota>,
            env: CastingEnvironment
        ): List<Iota> = listOf(args.getList(0, argc).find { it is IotaDuck && it.inscription != null && Iota.tolerates(it.inscription, args[1]) }.orNull())
    }

    val functor = object: ConstMediaAction {
        override val argc = 2

        override fun execute(
            args: List<Iota>,
            env: CastingEnvironment
        ): List<Iota> {
            val (target, label) = args
            check(target is IotaDuck)
            // This feels evil x2
            val data = IotaType.serialize(target)
            if (label is NullIota) {
                data.remove(FUNCTOR_KEY)
            } else {
                data.put(FUNCTOR_KEY, IotaType.serialize(label))
            }
            return listOf(IotaType.deserializeIota(data, env.world))
        }
    }

    val silence = object: SpellAction {
        override val argc = 1

        override fun execute(args: List<Iota>, env: CastingEnvironment): SpellAction.Result {
            val player = args.getPlayer(0, argc)
            val state = player.getComponent(playerStateComponent)
            return SpellAction.Result(object: RenderedSpell {
                override fun cast(env: CastingEnvironment) {
                    state.silence = true
                    state.silenceCause = env.castingEntity?.uuid?.toKotlinUuid() ?: Uuid.NIL
                    playerStateComponent.sync(player)
                }
            }, if (state.silence) 0 else MediaConstants.QUENCHED_BLOCK_UNIT, listOf())
        }
    }

    val unsilence = object: SpellAction {
        override val argc = 1

        override fun execute(args: List<Iota>, env: CastingEnvironment): SpellAction.Result {
            val player = args.getPlayer(0, argc)
            val state = player.getComponent(playerStateComponent)
            return SpellAction.Result(object: RenderedSpell {
                override fun cast(env: CastingEnvironment) {
                    state.silence = false
                    state.silenceCause = Uuid.NIL
                    playerStateComponent.sync(player)
                }
            }, if (!state.silence) 0 else if (env.castingEntity?.uuid?.toKotlinUuid() == state.silenceCause) MediaConstants.SHARD_UNIT * 3 else MediaConstants.QUENCHED_BLOCK_UNIT * 32, listOf())
        }
    }

    val createPocket = object: ConstMediaAction {
        override val argc = 0
        override val mediaCost = 8 * MediaConstants.QUENCHED_BLOCK_UNIT
        override fun execute(
            args: List<Iota>,
            env: CastingEnvironment
        ): List<Iota> {
            val name = Identifier(modid, "pocket/${UUID.randomUUID()}")
            val key = RegistryKey.of(RegistryKeys.WORLD, name)
            // this works for now
            val iota = PocketDimIota(key, env.world.registryManager, PocketDimInfo(pigment = env.pigment, worldSeed = env.world.random.nextLong()))
            iota.allocate(env.world.server) // TODO: lazy allocation
            print("mlem $iota")
            return listOf(iota)
        }
    }

    val box = object: ConstMediaAction {
        override val argc = 1
        override fun execute(
            args: List<Iota>,
            env: CastingEnvironment
        ) = listOf(BoxIota(IotaType.serialize(args[0])))
    }

    val unbox = object: ConstMediaAction {
        override val argc = 1
        override fun execute(
            args: List<Iota>,
            env: CastingEnvironment
        ): List<Iota> {
            val (box) = args
            if (box !is BoxIota) {
                throw MishapInvalidIota(box, 0, BoxIota.typeName())
            }
            if (box.data.getBoolean("pipebomb")) {
                env.world.createExplosion(null, DamageSource(env.world.registryManager[RegistryKeys.DAMAGE_TYPE].getEntry(pipeBombDamageTypeKey).orElseThrow()), object: ExplosionBehavior() {
                    override fun canDestroyBlock(
                        explosion: Explosion?,
                        world: BlockView?,
                        pos: BlockPos?,
                        state: BlockState?,
                        power: Float
                    ) = false
                }, env.castingEntity?.pos ?: throw MishapBadCaster(), 3.0f, false, World.ExplosionSourceType.TNT)
            }
            return listOf(IotaType.deserializeIota(box.data, env.world))
        }
    }
}

data class BoxIota(val data: NbtCompound): Iota(BoxIota, data) {
    override fun isTruthy() = true
    override fun toleratesOther(p0: Iota) = p0 is BoxIota && data == p0.data
    override fun serialize() = data
    companion object: IotaType<BoxIota>() {
        override fun deserialize(
            p0: NbtElement,
            p1: ServerWorld?
        ) = BoxIota(p0 as? NbtCompound ?: NbtCompound())

        override fun display(p0: NbtElement?): Text {
            if (p0 is NbtCompound) {
                if (p0.getBoolean("pipebomb")) {
                    return Text.translatable("Boxed Pipe Bomb").blue
                } else {
                    val type = getTypeFromTag(p0)
                    if (type != null) {
                        return Text.translatable("Boxed %s", type.typeName().copyContentOnly()).blue
                    }
                }
            }
            return Text.translatable("Box").blue
        }
        override fun color() = Formatting.BLUE.colorValue!!

        val pipeBomb = BoxIota(NbtCompound().also { it.putBoolean("pipebomb", true) })
    }
}

@Suppress("INAPPLICABLE_JVM_NAME")
interface MinecraftServerDuck {
    val executor: Executor
        @JvmName("hexmu\$getExecutor") get
    val session: LevelStorage.Session
        @JvmName("hexmu\$getSession") get
    var worlds: Map<RegistryKey<World>, ServerWorld>
        @JvmName("hexmu\$getWorlds") get
        @JvmName("hexmu\$setWorlds") set
}

val pipeBombDamageTypeKey: RegistryKey<DamageType> = RegistryKey.of(RegistryKeys.DAMAGE_TYPE, id("pipe_bomb"))

data class PocketDimIota(val key: RegistryKey<World>, val manager: DynamicRegistryManager, val info: PocketDimInfo): Iota(PocketDimIota, key to info) {
    override fun isTruthy() = manager[RegistryKeys.WORLD].contains(key)
    override fun toleratesOther(p0: Iota) = p0 is PocketDimIota && key.value == p0.key.value
    override fun serialize() = NbtCompound().apply {
        putString("key", key.value.toString())
        putCompound("info", info.toNBT())
    }

    fun allocate(server: MinecraftServer) {
        require(server.isOnThread) { "Pockets may only be allocated on the server thread" }
        require(server is MinecraftServerDuck) { "Server is not a duck." }
        val reg = manager[RegistryKeys.WORLD]
        if (reg[key] == null) {
            val worldBorder = server.overworld.worldBorder
            val serverLevelData = server.saveProperties.mainWorldProperties
            val world = ServerWorld(
                server,
                server.executor,
                server.session,
                UnmodifiableLevelProperties(server.saveProperties, serverLevelData),
                key,
                manager[RegistryKeys.DIMENSION][id("pocket")],
                object : WorldGenerationProgressListener {
                    override fun start(spawnPos: ChunkPos?) {}
                    override fun setChunkStatus(pos: ChunkPos?, status: ChunkStatus?) {}
                    override fun start() {}
                    override fun stop() {}
                },
                false,
                info.worldSeed,
                listOf(),
                false,
                server.overworld.randomSequences
            )
            worldBorder.addListener(WorldBorderListener.WorldBorderSyncer(world.worldBorder))
            server.worlds = server.worlds.toMutableMap().also { it[key] = world }
            worldBorder.load(serverLevelData.worldBorder)
            // TODO: dimension refresh
        }
    }

    companion object: IotaType<PocketDimIota>() {
        override fun deserialize(
            p0: NbtElement,
            p1: ServerWorld
        ): PocketDimIota? {
            println("awawa")
            if (p0 !is NbtCompound) return null
            val key = Identifier.tryParse(p0.getString("key")) ?: return null
            val info = PocketDimInfo()
            println("mrrow :3")
            info.fromNBT(p0.getCompound("info"))
            println("mrrp")
            return PocketDimIota(
                RegistryKey.of(RegistryKeys.WORLD, key),
                p1.registryManager,
                info,
            )
        }

        val timeValue = sin(System.currentTimeMillis() / 10000.0).toFloat()

        override fun display(p0: NbtElement): Text {
            if (p0 is NbtCompound) {
                val key = p0.getString("key").takeLast(6)
                val text = Text.translatable("[Pocket %s]", Text.literal(key).styled { it.withFont(Identifier("minecraft", "illageralt")) })
                val pigment = p0.getCompound("info").getCompound("pigment")
                if (!pigment.isEmpty) text.styled { it.withColor(FrozenPigment.fromNBT(pigment).colorProvider.getColor(timeValue, Vec3d.ZERO)) }
                return text
            }
            return Text.literal("[Pocket]")
        }

        override fun color() = FrozenPigment.DEFAULT.get().colorProvider.getColor(timeValue, Vec3d.ZERO)
    }
}

object MediaStrand: Item(Settings()), IotaHolderItem {
    override fun readIotaTag(p0: ItemStack) = p0.nbt!!.getCompound("iota")!!
    override fun writeable(p0: ItemStack) = false
    override fun canWrite(p0: ItemStack, p1: Iota?) = false
    override fun writeDatum(p0: ItemStack, p1: Iota?) {}
    fun of(iota: Iota) = ItemStack(MediaStrand, 1).also {
        it.orCreateNbt.putCompound("iota", IotaType.serialize(iota))
    }
}

data class PocketDimInfo(var pigment: FrozenPigment = FrozenPigment.DEFAULT.get(), var worldSeed: Long = 0L) {
    fun fromNBT(p0: NbtCompound) {
        if (p0.contains("pigment", NbtElement.COMPOUND_TYPE.toInt())) {
            pigment = FrozenPigment.fromNBT(p0.getCompound("pigment"))
        } else {
            pigment = FrozenPigment.DEFAULT.get()
        }
    }

    fun toNBT() = NbtCompound().apply {
        putCompound("pigment", pigment.serializeToNBT())
    }
}

data class PocketDimMeta(val world: World, val info: PocketDimInfo = PocketDimInfo()): Component, AutoSyncedComponent {
    override fun readFromNbt(p0: NbtCompound) {
        info.fromNBT(p0)
    }

    override fun writeToNbt(p0: NbtCompound) {
        p0.copyFrom(info.toNBT())
    }
}

@Suppress("INAPPLICABLE_JVM_NAME")
interface IotaDuck {
    var inscription: Iota?
        @JvmName("hexmu\$getInscription") get
        @JvmName("hexmu\$setInscription") set
    var functor: Iota?
        @JvmName("hexmu\$getFunctor") get
        @JvmName("hexmu\$setFunctor") set
    var expiry: Long
        @JvmName("hexmu\$getExpiry") get
        @JvmName("hexmu\$setExpiry") set
    var annotations: MutableMap<Iota, Iota>
        @JvmName("hexmu\$getAnnotations") get
        @JvmName("hexmu\$setAnnotations") set
    var server: MinecraftServer
        @JvmName("hexmu\$getServer") get
        @JvmName("hexmu\$setServer") set
}

fun id(value: String) = Identifier(modid, value)
data class PlayerState
@JvmOverloads constructor(
    val player: PlayerEntity,
    var uwuify: Boolean = false,
    var uwuifyCause: Uuid = Uuid.NIL,
    var silence: Boolean = false,
    var silenceCause: Uuid = Uuid.NIL,
): Component, AutoSyncedComponent {
    override fun readFromNbt(p0: NbtCompound) {
        uwuify = p0.getBoolean("uwuify")
        uwuifyCause = p0.getUUID("uwuifyCause")?.toKotlinUuid() ?: Uuid.NIL
        silence = p0.getBoolean("silence")
        silenceCause = p0.getUUID("silenceCause")?.toKotlinUuid() ?: Uuid.NIL
    }

    override fun writeToNbt(p0: NbtCompound) {
        p0.putBoolean("uwuify", uwuify)
        p0.putUuid("uwuifyCause", uwuifyCause.toJavaUuid())
        p0.putBoolean("silence", silence)
        p0.putUuid("silenceCause", silenceCause.toJavaUuid())
    }
}

val playerStateComponent: ComponentKey<PlayerState> by lazy {
    ComponentRegistry.getOrCreate(id("player_state"), PlayerState::class.java)
}
val pocketDataComponent: ComponentKey<PocketDimMeta> by lazy {
    ComponentRegistry.getOrCreate(id("pocket_data"), PocketDimMeta::class.java)
}
fun init() {
    Registry.register(Registries.BLOCK, id("filler"), FillerBlock)
    Registry.register(Registries.ITEM, id("iota"), MediaStrand)
    Registry.register(hexActions, id("uwuify"), ActionRegistryEntry(Patterns.uwuify, Actions.uwuify))
    Registry.register(hexActions, id("inscribe"), ActionRegistryEntry(Patterns.inscribe, Actions.inscribe))
    Registry.register(hexActions, id("inscription"), ActionRegistryEntry(Patterns.inscription, Actions.inscription))
    Registry.register(hexActions, id("substring"), ActionRegistryEntry(Patterns.search, Actions.search))
    Registry.register(hexActions, id("functor"), ActionRegistryEntry(Patterns.functor, Actions.functor))
    //Registry.register(hexActions, id("pocket/open"), ActionRegistryEntry(Patterns.conjurePocket, Actions.createPocket))
    //Registry.register(hexActions, id("pocket/close"), ActionRegistryEntry(Patterns.dismissPocket, Actions.functor))
    Registry.register(hexActions, id("silence"), ActionRegistryEntry(Patterns.silence, Actions.silence))
    Registry.register(hexActions, id("unsilence"), ActionRegistryEntry(Patterns.unsilence, Actions.unsilence))
    Registry.register(hexActions, id("box"), ActionRegistryEntry(Patterns.boxIota, Actions.box))
    Registry.register(hexActions, id("unbox"), ActionRegistryEntry(Patterns.unboxIota, Actions.unbox))
    Registry.register(hexActions, id("pipebomb"), ActionRegistryEntry(Patterns.boxPipebomb, Action.makeConstantOp(BoxIota.pipeBomb)))
    Registry.register(hexActions, id("annotation/get"), ActionRegistryEntry(Patterns.annotate, object: ConstMediaAction {
        override val argc = 3
        override fun execute(args: List<Iota>, env: CastingEnvironment): List<Iota> {
            val (target, key, value) = args
            val copy = IotaType.deserializeIota(IotaType.serialize(target), env.world)
            check(copy is IotaDuck)
            println("annotating $target (copied to $copy) with $key → $value")
            val realKey = copy.annotations.keys.find {
                val r = Iota.tolerates(it, key)
                println("Is $it a key? $r")
                r
            } ?: key.also { println("no realKey") }
            if (value is NullIota) {
                copy.annotations.remove(realKey)
            } else {
                copy.annotations[key] = value
            }
            return listOf(copy)
        }
    }))
    Registry.register(hexActions, id("annotation/set"), ActionRegistryEntry(Patterns.readAnnotation, object: ConstMediaAction {
        override val argc = 2
        override fun execute(args: List<Iota>, env: CastingEnvironment) = listOf((args[0] as? IotaDuck)?.annotations?.let { it[args[1]] ?: it.entries.find { Iota.tolerates(args[1], it.key) }?.value }.orNull())
    }))
    Registry.register(xplat.iotaTypeRegistry, id("pocket"), PocketDimIota)
    Registry.register(xplat.iotaTypeRegistry, id("box"), BoxIota)
}

val Iota.duck get() = this as IotaDuck

//object ConfigManager {
//    lateinit var configScript
//    lateinit var configObject
//    fun reload() {
//        configScript = compileJS()
//    }
//}

fun EntityComponentFactoryRegistry.initComponents() {
    registerForPlayers(playerStateComponent, ::PlayerState, RespawnCopyStrategy.CHARACTER)
}

fun WorldComponentFactoryRegistry.initWorldComponents() {
    register(pocketDataComponent, ::PocketDimMeta)
}

val jsContext = Context().apply {
    languageVersion = Context.VERSION_ES6
    wrapFactory = object : WrapFactory() {
        override fun wrap(
            cx: Context,
            scope: Scriptable,
            obj: Any?,
            staticType: Class<*>
        ): Any? =
            when (obj) {
                null, is String, is Number, is Boolean -> obj
                is Char -> obj.toString()
                else -> super.wrap(cx, scope, obj, staticType)
            }
    }
}

fun compileJS(code: String, name: String = "unknown", line: Int = 1): Script {
    // what drunkard designed this API??
    Context.enter(jsContext)
    try {
        return jsContext.compileString(code, name, line, null)
    } finally {
        Context.exit()
    }
}

fun compileJS(code: Reader, name: String = "unknown", line: Int = 1): Script {
    Context.enter(jsContext)
    try {
        return jsContext.compileReader(code, name, line, null)
    } finally {
        Context.exit()
    }
}

fun Script.execWithGlobals(globals: Map<String, Any>): Any {
    Context.enter(jsContext)
    try {
        val scope = jsContext.initStandardObjects()
        globals.forEach {
            scope.put(it.key, scope, it.value)
        }
        return exec(jsContext, scope)
    } finally {
        Context.exit()
    }
}

fun invokeWithFunctor(
    vm: CastingVM,
    world: ServerWorld,
    cont: SpellContinuation,
    functor: Iota,
): CastResult {
    val (image, effects, newCont, sound) = OpEval.exec(vm.env, vm.image, cont, vm.image.stack.toMutableList(), functor)
    return CastResult(
        functor,
        newCont,
        image,
        effects,
        ResolvedPatternType.EVALUATED,
        sound,
    )
}

class HexPatternBuilder(val startDir: HexDir) {
    var angles = mutableListOf<HexAngle>()

    companion object {
        val northeast get() = HexPatternBuilder(HexDir.NORTH_EAST)
        val east get() = HexPatternBuilder(HexDir.EAST)
        val southeast get() = HexPatternBuilder(HexDir.SOUTH_EAST)
        val southwest get() = HexPatternBuilder(HexDir.SOUTH_WEST)
        val west get() = HexPatternBuilder(HexDir.WEST)
        val northwest get() = HexPatternBuilder(HexDir.NORTH_WEST)
    }

    val a get() = also { angles += HexAngle.LEFT_BACK }
    val q get() = also { angles += HexAngle.LEFT }
    val w get() = also { angles += HexAngle.FORWARD }
    val e get() = also { angles += HexAngle.RIGHT }
    val d get() = also { angles += HexAngle.RIGHT_BACK }
    val s get() = also { angles += HexAngle.BACK }

    operator fun invoke() = HexPattern(startDir, angles.toMutableList())
}

@Suppress("INAPPLICABLE_JVM_NAME")
interface RawIotaHolderItem: IotaHolderItem {
    var ItemStack.iotaTag: NbtCompound?
        @JvmName("readIotaTag") @Suppress("ACCIDENTAL_OVERRIDE") get
        @JvmName("hexmu\$writeIotaTag") set
}

fun copyFromBookself(
    shelf: BlockEntityAkashicBookshelf,
    world: World,
    pos: BlockPos,
    player: PlayerEntity,
    stack: ItemStack,
    cir: CallbackInfoReturnable<ActionResult>
) {
    val item = stack.item
    if (item is IotaHolderItem) {
        if (shelf.iotaTag == null || !item.writeable(stack)) {
            cir.returnValue = ActionResult.FAIL
            return
        }
        if (item is RawIotaHolderItem) {
            if (!world.isClient) {
                with(item) {
                    stack.iotaTag = shelf.iotaTag
                }
            }
            world.playSound(player, pos, HexSounds.SCROLL_SCRIBBLE, SoundCategory.BLOCKS, 1f, 1f);
            cir.returnValue = ActionResult.SUCCESS
            return
        }
        // we can only hunch
        if (world is ServerWorld) {
            val iota = IotaType.deserializeIota(shelf.iotaTag, world)
            if (item.canWrite(stack, iota)) {
                item.writeDatum(stack, iota);
                world.playSound(null, pos, HexSounds.SCROLL_SCRIBBLE, SoundCategory.BLOCKS, 1f, 1f);
            }
        }
        cir.returnValue = ActionResult.SUCCESS
    }
}
