package org.eu.net.pool.common_curses

import net.fabricmc.fabric.api.entity.event.v1.ServerLivingEntityEvents
import net.fabricmc.fabric.api.entity.event.v1.ServerPlayerEvents
import net.fabricmc.fabric.api.event.EventFactory
import net.fabricmc.fabric.api.event.lifecycle.v1.ServerEntityEvents
import net.fabricmc.fabric.api.event.lifecycle.v1.ServerTickEvents
import net.fabricmc.fabric.api.networking.v1.EntityTrackingEvents
import net.minecraft.entity.player.PlayerEntity
import net.minecraft.item.ItemStack
import net.minecraft.text.OrderedText
import net.minecraft.text.StringVisitable
import net.minecraft.text.Style
import org.eu.net.pool.common_curses.HotbarRendering.ALL
import org.mozilla.javascript.Context
import org.mozilla.javascript.Script
import org.mozilla.javascript.Scriptable
import org.mozilla.javascript.WrapFactory
import java.io.Reader
import java.util.Optional
import kotlin.code
import kotlin.collections.unzip
import kotlin.text.forEach

inline fun <reified T: Any> createEvent(crossinline merge: Array<T>.() -> T) = EventFactory.createArrayBacked<T>(T::class.java) { it.merge() }!!

enum class SlotAccess(val canInsert: Boolean, val canRemove: Boolean, val dropItems: Boolean) {
    ALLOW(true, true, false),
    INSERT_ONLY(true, false, false),
    REMOVE_ONLY(false, true, false),
    LOCK(false, false, false),
    LOCK_AND_DROP(false, false, true);
    companion object {
        @JvmField val playerInventory = createEvent<(PlayerEntity, Int, ItemStack) -> SlotAccess> {
            { p, s, i -> map { it(p, s, i) }.merge() }
        }
    }
}

val jsContext = Context().apply {
    languageVersion = Context.VERSION_ES6
    wrapFactory = object : WrapFactory() {
        override fun wrap(
            cx: Context,
            scope: Scriptable,
            obj: Any?,
            staticType: Class<*>
        ): Any? =
            when (obj) {
                null, is String, is Number, is Boolean -> obj
                is Char -> obj.toString()
                else -> super.wrap(cx, scope, obj, staticType)
            }
    }
}

fun compileJS(code: String, name: String = "unknown", line: Int = 1): Script {
    // what drunkard designed this API??
    Context.enter(jsContext)
    try {
        return jsContext.compileString(code, name, line, null)
    } finally {
        Context.exit()
    }
}

fun compileJS(code: Reader, name: String = "unknown", line: Int = 1): Script {
    Context.enter(jsContext)
    try {
        return jsContext.compileReader(code, name, line, null)
    } finally {
        Context.exit()
    }
}

fun Script.execWithGlobals(globals: Map<String, Any>): Any {
    Context.enter(jsContext)
    try {
        val scope = jsContext.initStandardObjects()
        globals.forEach {
            scope.put(it.key, scope, it.value)
        }
        return exec(jsContext, scope)
    } finally {
        Context.exit()
    }
}

interface TextManipulator {
    val priority get() = 0
    fun modify(text: String, atEnd: Boolean = true): String
    fun modify(text: OrderedText, atEnd: Boolean = true) = OrderedText { f ->
        var list = mutableListOf<Pair<Style, MutableList<Int>>>()
        text.accept { idx, style, char ->
            if (list.isNotEmpty() && list.last().first == style)
                list.last().second.add(char)
            else
                list.add(style to mutableListOf(char))
        }
        list.forEachIndexed { i, (style, points) ->
            modify(String(points.map(Int::toChar).toCharArray()), atEnd && i == list.lastIndex).forEachIndexed { i, it ->
                if (!f.accept(i, style, it.code)) return@OrderedText false
            }
        }
        true
    }
    fun modify(text: StringVisitable, atEnd: Boolean = true) = object: StringVisitable {
        override fun <T> visit(visitor: StringVisitable.Visitor<T>): Optional<T> =
            text.visit {
                visitor.accept(modify(it, false))
            }.or {
                visitor.accept(modify("", true))
            }
        override fun <T> visit(styledVisitor: StringVisitable.StyledVisitor<T>, style: Style): Optional<T> =
            text.visit({ sty, it ->
                styledVisitor.accept(sty, modify(it, false))
            }, style).or {
                styledVisitor.accept(style, modify("", true))
            }
    }

    object None: TextManipulator {
        override val priority get() = -Int.MIN_VALUE
        override fun modify(text: String, atEnd: Boolean) = text
    }
    object UwU: TextManipulator {
        override fun modify(text: String, atEnd: Boolean) = text.uwuify(atEnd)
    }
    object Zephyr: TextManipulator {
        override fun modify(text: String, atEnd: Boolean) = if (atEnd) "$text qwq" else text
    }
    object Empty: TextManipulator {
        override val priority get() = 2000
        override fun modify(text: String, atEnd: Boolean) = ""
    }
}

private val UWUIFY_PHRASES = arrayOf(
    "UwU",
    "owo",
    "OwO",
    "uwu",
    ">w<",
    "^w^",
    ":3",
    "^-^",
    "^_^",
    "^w^",
    ":3"
)

// Copied with modifications from https://github.com/MayaqqDev/Cynosure/blob/master/common/src/main/kotlin/dev/mayaqq/cynosure/utils/fun/UwUfy.kt
// Maya uses Mojmap, so a proper dependency is not an option
fun String.uwuify(addPhrase: Boolean = true): String {
    var input = this
    if (!any(Char::isLetter)) return input
    val stringLength = input.length
    // Replace 'r' and 'l' with 'w', and 'R' and 'L' with 'W'
    // Replace 'ove' with 'uv' and 'OVE' with 'UV'
    // Replace 'o' with 'owo' and 'O' with 'OwO'
    // Replace repeated exclamation marks and question marks
    input = input
        .replace("[rl]".toRegex(), "w").replace("[RL]".toRegex(), "W")
        .replace("ove".toRegex(), "uv").replace("OVE".toRegex(), "UV")
        .replace("o".toRegex(), "owo").replace("O".toRegex(), "OwO")
        .replace("!".toRegex(), "!!!").replace("\\?".toRegex(), "???")

    // Convert to uppercase
    if (stringLength % 3 == 0) {
        input = input.uppercase()
    }

    input = input.replace(Regex("%(\\p{L})")) { m -> "%" + m.groupValues[1].lowercase() }
    input = input.replace(Regex("\\$(\\p{L})")) { m -> "\\$" + m.groupValues[1].lowercase() }

    input = if (stringLength % 2 == 0) {
        // Add more letters to the end of words (Not numbers!).minByOrNull(HotbarRendering::ordinal) ?: ALL
        input.replace("(\\p{L})(\\b)".toRegex(), "$1$1$1$1$2")
    } else {
        // 50% chance to duplicate the first letter and add '-'
        input.replace("\\b(\\p{L})(\\p{L}*)\\b".toRegex(), "$1-$1$2")
    }

    if (addPhrase) input += " " + UWUIFY_PHRASES[stringLength % UWUIFY_PHRASES.size]
    return input
}

fun init() {
    
}

/**
 * Fired to control hotbar slots' visibility. The most severe rendering change (the one with the highest [ordinal][HotbarRendering.ordinal]) is used.
 */
enum class HotbarRendering {
    /**
     * The player's entire hotbar is hidden. Mods using this should ensure that the offhand is inaccessible, as the behavior of offhand items in this mode is undefined.
     */
    NONE,

    /**
     * Only the player's [selectedSlot][net.minecraft.entity.player.PlayerInventory.selectedSlot] is visible. Mods using this should ensure that if they want to only render one slot, the selected slot is locked in some way. Common Curses currently provides no selected-slot-locking behavior.
     */
    SELECTED_SLOT,

    /**
     * Follow vanilla hotbar rendering behavior. All slots are visible.
     */
    ALL;
    companion object {
        val event = createEvent<() -> HotbarRendering> { { map { it() }.merge() } }
    }
}

inline fun <T, T1, U> Pair<T, U>.map1(f: (T) -> T1) = f(first) to second
inline fun <T, U, U1> Pair<T, U>.map2(f: (U) -> U1) = first to f(second)

fun List<SlotAccess>.merge() = fold(Triple(true, true, false)) { t, s -> t.copy(t.first && s.canInsert, t.second && s.canRemove, t.third && s.dropItems) }.run {
    if (third)
        SlotAccess.LOCK_AND_DROP
    else if (first && second)
        SlotAccess.ALLOW
    else if (first)
        SlotAccess.INSERT_ONLY
    else if (second)
        SlotAccess.REMOVE_ONLY
    else
        SlotAccess.LOCK
}
fun List<TextManipulator>.merge() = object: TextManipulator {
    override val priority: Int get() = maxOf(TextManipulator::priority)
    override fun modify(text: String, atEnd: Boolean) = fold(text) { t, m -> m.modify(t, atEnd) }
    override fun modify(text: OrderedText, atEnd: Boolean) = fold(text) { t, m -> m.modify(t, atEnd) }
}
fun List<HotbarRendering>.merge() = minByOrNull(HotbarRendering::ordinal) ?: ALL