package io.github.tropheusj.serialization_hooks.ingredient;

import com.google.gson.JsonElement;

import com.google.gson.JsonObject;

import com.google.gson.JsonSyntaxException;

import io.github.tropheusj.serialization_hooks.SerializationHooks;
import io.github.tropheusj.serialization_hooks.ingredient.CombinedIngredient.Deserializer;
import io.netty.handler.codec.DecoderException;
import net.fabricmc.fabric.api.event.registry.FabricRegistryBuilder;
import net.fabricmc.fabric.api.event.registry.RegistryAttribute;
import net.minecraft.class_1856;
import net.minecraft.class_2378;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;

/**
 * An IngredientDeserializer handles turning packets and Json back into actual Ingredients.
 */
public interface IngredientDeserializer {
	/**
	 * The Registry for deserializers.
	 */
	class_2378<IngredientDeserializer> REGISTRY = FabricRegistryBuilder.createSimple(
			IngredientDeserializer.class, SerializationHooks.id("ingredient_deserializers")
	).attribute(RegistryAttribute.SYNCED).buildAndRegister();

	/**
	 * List of deserializer IDs that are referenced in recipes but were not found.
	 * Used to avoid log spam.
	 */
	List<class_2960> KNOWN_MISSING = new ArrayList<>();

	/**
	 * Create an Ingredient from the packet.
	 * This should reflect the corresponding {@link class_1856#method_8088(class_2540)} method in your Ingredient.
	 */
	class_1856 fromNetwork(class_2540 buffer);

	/**
	 * Create an Ingredient from the given json object.
	 * This should reflect the corresponding {@link class_1856#method_8089()} method in your Ingredient.
	 */
	class_1856 fromJson(JsonObject object);

	static void init() {
		// load the class and registry
		class_2378.method_10230(REGISTRY, Deserializer.ID, Deserializer.INSTANCE);
	}

	/**
	 * Try to deserialize an Ingredient from the given JsonObject.
	 * @return the deserialized ingredient, or null if not custom
	 */
	@Nullable
	static class_1856 tryDeserializeJson(JsonObject object) {
		JsonElement type = object.get("type");
		if (type != null && type.isJsonPrimitive()) {
			class_2960 id = class_2960.method_12829(type.getAsString());
			if (id == null)
				return null;
			IngredientDeserializer deserializer = IngredientDeserializer.REGISTRY.method_10223(id);
			if (deserializer != null) {
				try {
					return deserializer.fromJson(object);
				} catch (JsonSyntaxException ex) {
					SerializationHooks.LOGGER.error("Failed to deserialize Ingredient using deserializer [{}]: {}", id, ex.getMessage());
				}
			}
			if (KNOWN_MISSING.contains(id))
				return null;
			KNOWN_MISSING.add(id);
			SerializationHooks.LOGGER.error("IngredientDeserializer with ID not found: [{}] this can be ignored unless issues occur.", id);
		}
		return null;
	}

	/**
	 * Try to deserialize an Ingredient from the given buffer.
	 * @return the deserialized ingredient, or null if not custom
	 */
	@Nullable
	static class_1856 tryDeserializeNetwork(class_2540 buf) {
		int readIndex = buf.readerIndex();
		try {
			class_2960 id = class_2960.method_12829(buf.method_19772());
			if (id != null && !id.method_12832().isEmpty()) {
				IngredientDeserializer deserializer = IngredientDeserializer.REGISTRY.method_10223(id);
				if (deserializer != null)
					return deserializer.fromNetwork(buf);
				if (KNOWN_MISSING.contains(id))
					return null;
				KNOWN_MISSING.add(id);
				SerializationHooks.LOGGER.error("IngredientDeserializer with ID not found: [{}] this can be ignored unless issues occur.", id);
			}
			buf.readerIndex(readIndex);
			return null;
		} catch (DecoderException e) { // not a string
			buf.readerIndex(readIndex);
			return null;
		}
	}
}
